/* lighTS - An extensible and lightweight Linda-like tuplespace
 * Copyright (C) 2001, Gian Pietro Picco
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *  
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package lights.interfaces;

/**
 * Defines the operations available on a tuple space. The traditional operations
 * provided by Linda, namely the insertion of a tuple with <code>out</code>
 * and pattern matching with the blocking operations <code>in</code> and
 * <code>rd</code>, are supported. In addition, the following features are
 * provided:
 * <UL>
 * <LI>non-blocking <I>probe</I> operations <code>inp</code> and
 * <code>rdp</code>;
 * <LI><I>group</I> operations <code>outg</code>, <code>ing</code>,
 * <code>rdg</code>, that allow to write and retrieve multiple tuples at
 * once;
 * <LI>a <code>count</code> operation returning the number of tuples matching
 * a given templates;
 * <LI>a name associated to the tuple space.
 * </ul>
 * 
 * @author <a href="mailto:picco@elet.polimi.it">Gian Pietro Picco</a>
 * @version 1.0
 */

public interface ITupleSpace  {

	/**
	 * The default name of a tuple space.
	 */

	public static final String DEFAULT_NAME = "_aTupleSpace_";

	/**
	 * Returns the name of the tuple space.
	 */

	public String getName();

	/**
	 * Inserts a tuple in the tuple space. Note how different implementations
	 * may specify different atomicity requirements.
	 * 
	 * @param tuple
	 *            The tuple to be inserted.
	 * @exception TupleSpaceException
	 *                if an error occurs in the implementation.
	 */

	public void out(ITuple tuple) throws TupleSpaceException;

	/**
	 * Inserts multiple tuples in the tuple space. Note how different
	 * implementations may specify different atomicity requirements.
	 * 
	 * @param tuples
	 *            An array containing the tuples to be inserted.
	 * @exception TupleSpaceException
	 *                if an error occurs in the implementation.
	 */

	public void outg(ITuple[] tuples) throws TupleSpaceException;

	/**
	 * Withdraws from the tuple space a tuple matching the templates specified;
	 * if no tuple is found, the caller is suspended until such a tuple shows up
	 * in the tuple space. Note how different implementations may specify
	 * different policies for the selection of the tuple returned among multiple
	 * matches.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return a tuple matching the templates.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple in(ITuple template) throws TupleSpaceException;

	/**
	 * Withdraws from the tuple space a tuple matching the templates specified;
	 * if no tuple is found, <code>null</code> is returned. Note how different
	 * implementations may specify different policies for the selection of the
	 * tuple returned among multiple matches.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return a tuple matching the templates, or <code>null</code> if none is
	 *         found.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple inp(ITuple template) throws TupleSpaceException;

	/**
	 * Withdraws from the tuple space <i>all</I> the tuple matching the
	 * templates specified. If no tuple is found, <code>null</code> is
	 * returned.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return a tuple matching the templates, or <code>null</code> if none is
	 *         found.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple[] ing(ITuple template) throws TupleSpaceException;

	/**
	 * Reads from the tuple space a copy of a tuple matching the templates
	 * specified. If no tuple is found, the caller is suspended until such a
	 * tuple shows up in the tuple space. Note how different implementations may
	 * specify different policies for the selection of the tuple returned among
	 * multiple matches.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return a copy of a tuple matching the templates.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple rd(ITuple template) throws TupleSpaceException;

	/**
	 * Returns a count of the tuples found in the tuple space that match the
	 * templates.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return the number of tuples currently in the tuple space that match the
	 *         templates.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */
	
	/**
	 * Reads from the tuple space a copy of a tuple matching the templates
	 * specified. If no tuple is found, <code>null</code> is returned. Note
	 * how different implementations may specify different policies for the
	 * selection of the tuple returned among multiple matches.
	 * 
	 * @param templates
	 *            the templates used for matching.
	 * @return a copy of a tuple matching the templates.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple rdp(ITuple template) throws TupleSpaceException;

	/**
	 * Reads from the tuple space a copy of <I>all</I> the tuples matching the
	 * templates specified. If no tuple is found, <code>null</code> is
	 * returned.\
	 * 
	 * @param template
	 *            the template used for matching.
	 * @return a copy of a tuple matching the templates.
	 * @exception TupleSpaceException
	 *                if an error in the implementation.
	 */

	public ITuple[] rdg(ITuple template) throws TupleSpaceException;

	/**
	 * Reads from the tuple space all the tuples matching the input tuple, and returns the number of such tuples.
	 * @param template the template used for matching
	 * @return the number of tuples matching the template
	 * @throws TupleSpaceException
	 */
	public int count(ITuple template) throws TupleSpaceException;

}
