/*
 * Queen.java
 *
 * history:
 * 2001.05.03 alokem creation
 */

import magenta.*;
import java.util.*;

/**
 * class Queen -
 *  a magenta manager that implements the behaviour of a chess queen.
 *  This class participates with other Queens and the Nqueens server
 *  to attempt a distributed solution to the n-queens problem.
 *
 *  <p>
 *  The queen has the following behaviour:
 *  <ul>
 *  <li>queen does not wake up until it has valid hostinfo and
 *  serverinfo (hostname:port of nqueens object)
 *  <li>queen queries nqueens for list of known queens and adds
 *  them as observers, queen adds itself to nqueens list of queens
 *  <li>queen makes a random move - observer queens are notified
 *  <li>the movements of other queens are received as events and
 *  the queen reacts accordingly
 *  <li>queen learns about new queens from the events it receives
 *  </ul>
 *
 *  <dl>
 *  <b>GdmoObject attributes:</b>
 *    <dd>HostInfo - </dd>
 *    <dd>ServerInfo - hostname:port of the nqueens server</dd>
 *    <dd>Position - co-ordinates of this queen on the board</dd>
 *  </dl>
 *
 * @author aloke mukherjee
 * @version
 * 2001.05.03 alokem creation
 *
 * FIXME
 * remove hardcoded value "nqueens-server"
 */
public class Queen extends magenta.Agent {
  /** flag to keep track of whether hostinfo attribute has been set */
  private boolean hostInfoSet = false;

  /** flag to keep track of whether serverinfo attribute has been set */
  private boolean serverInfoSet = false;

  /** hostname:port of nqueens server */
  private String serverInfo = "";

  /** number of rows/columns on board - FIXME there should be only one for both */
  final int MAX_SIZE = 4;

  /** which row the queen is on - zero based */
  private int row;

  /** which column the queen is on - zero based */
  private int column;

  /** slow-down factor - even if threatened don't move this many times */
  final int MAX_TRYAGAINS = 10;

  /** class state variable to keep track of how many times we stood our ground */
  private int tryAgainCount;

  /**
   * Queen::isSuccess -
   *  is the first token of the argument "success"?
   *  FIXME - should be a general magenta utility
   *
   * @param result-string string whose first token is either
   *                      "success" or "failure"
   *
   * @return true if the first token is "success"
   *
   * @version
   * 2001.05.03 alokem creation
   */
  private boolean
  isSuccess(String resultString) {
    StringTokenizer tokenizer = 
     new StringTokenizer(resultString, " "); 
    return (tokenizer.hasMoreElements() &&
            tokenizer.nextToken().compareTo("success") == 0);
  } // Queen::isSuccess

  /**
   * Queen::move -
   *  randomly pick a new column and row for the queen.
   *
   * @version
   * 2001.05.04 alokem creation
   */
  private void
  move() {
    String posString = new String((int)(MAX_SIZE * Math.random()) + "," + 
                                  (int)(MAX_SIZE * Math.random()));

    // causes event notifications to be sent to other queens
    getObjectManager().set("root*Position*" + posString);
  } // Queen::move

  /**
   * Queen::threatened -
   *  check if positiont threatens us and move out of the way.
   *
   * @param positionString string in form "row,column" indicating
   *                       position of another queen
   *
   * @version
   * 2001.05.04 alokem creation
   */
  private boolean
  threatened(String positionString) {
    int newRow, newColumn, theRow, theColumn;
    boolean isThreatened, isStillThreatened;
    String posString;

    // FIXME - factor this out of here and setposition

    // parse row and column from positionstring
    StringTokenizer tokenizer = new StringTokenizer(positionString, ",");
    if (tokenizer.hasMoreElements()) {
      newRow = Integer.parseInt(tokenizer.nextToken());
      if (newRow > MAX_SIZE) {
        return false;
      }
    } else {
      return false;
    }

    if (tokenizer.hasMoreElements()) {
      newColumn = Integer.parseInt(tokenizer.nextToken());
      if (newColumn > MAX_SIZE) {
        return false;
      }
    } else {
      return false;
    }

    // now check if it is in same column, row or diagonal
    isThreatened = ((newRow == row) || (newColumn == column) ||
                    (Math.abs(newRow - row) == Math.abs(newColumn - column)));
    isStillThreatened = isThreatened;
    
    while (isStillThreatened) {
      // move randomly until we find a position safe relative to this one
      theRow = (int)(MAX_SIZE * Math.random());
      theColumn = (int)(MAX_SIZE * Math.random());

      isStillThreatened = ((newRow == theRow) || (newColumn == theColumn) ||
                           (Math.abs(newRow - theRow) == Math.abs(newColumn - theColumn)));
      if (isStillThreatened == false) {
        tryAgainCount--;
        if (tryAgainCount == 0) {
          // contains the new safe location
          posString = new String(theRow + "," + theColumn);
          tryAgainCount = (int)(MAX_TRYAGAINS * Math.random());
        } else {
          // just stay in place
          posString = new String(row + "," + column);
        } // tryAgainCount == 0
        // causes event notifications to be sent to other queens
        getObjectManager().set("root*Position*" + posString);
      } // stillThreatened = false
    } // isstillthreatened

    return isThreatened;
  } // Queen::threatened

  /**
   * Queen::wakeUp -
   *  register queen with nqueens server,
   *  get list of known queens from the server, 
   *  add self to the list and make a move.
   *
   * @param isHostInfoSet   whether hostinfo is set
   * @param isServerInfoSet whether serverinfo is set -
   *                        both parms must be true
   *                        for wakeup to start!
   *
   * @version
   * 2001.05.03 alokem creation
   */
  private void
  wakeUp(boolean isHostInfoSet, boolean isServerInfoSet) {
    String getResult, queenHostInfo;
    String queenList = null;
    AgentProxy server;
    StringTokenizer queenTokenizer;

    // don't wake up until we now our hostname:port and the server's
    if (!isHostInfoSet || !isServerInfoSet) {
      return;
    }

    log(getName() + " has woken up");
  
    server = (AgentProxy)getObjectManager().getObject("root/observers/nqueens-server");

    // try to get QueenList from the server
    try {
      getResult = getComManager().send(server,
       "get*root*QueenList");
      log("queenList: " + getResult);
    } catch (Exception e) {
      // send wasn't successful because of server problem?
      // allow serverinfo to be changed
      serverInfoSet = false;
      return;
    }

    // get wasn't successful - server doesn't have queenlist?
    // allow serverinfo to be changed
    if (getResult.indexOf("get success") == -1) {
      serverInfoSet = false; 
      return;
    }

    // extract the list of queens from the get response
    // (response in form "get success object-name::attribute=value")
    queenTokenizer = 
     new StringTokenizer(getResult, "=");
    if (queenTokenizer.hasMoreElements()) {
      // discard first token
      queenTokenizer.nextToken();
      if (queenTokenizer.hasMoreElements()) {
        queenList = queenTokenizer.nextToken();
      }
    } // queenTokenizer has more elements

    // now parse the list of queens and add each queen into our observers list
    if (queenList != null) {
      queenTokenizer = new StringTokenizer(queenList, ",");

      while (queenTokenizer.hasMoreElements()) {
        queenHostInfo = queenTokenizer.nextToken();
        if (queenHostInfo.compareTo("(null)") != 0) {
          // create the queen with its name = hostinfo
          getObjectManager().create("magenta.AgentProxy*root/observers*" + queenHostInfo);
          getObjectManager().set("root/observers/" + queenHostInfo + "*HostInfo*" +
           queenHostInfo);
        } // ! empty list
      } // for each queen
    } // queenlist != null

    // add my hostname:port to the list of queen hostinfo on the nqueens server
    try {
      getComManager().send(server, 
       "set*root*QueenList*" + getHostInfo());
    } catch (Exception e) {
      log("could not add self (" + getHostInfo() + ") to QueenList" +
        " on server: " + server);
    }

    // move to a random position (this notifies other queens)
    move();
  } // Queen::wakeUp

  /**
   * Queen constructor -
   *  instantiate magenta components and useful objects.
   * 
   * @param path    path to object in object tree
   * @param name    name of object in object tree
   *
   * @version
   * 2001.05.03 alokem creation
   */
  public
  Queen(String path, String name) {
    super(path, name);

    // randomly init number of times to wait until making first move
    tryAgainCount = (int)(MAX_TRYAGAINS * Math.random());

    // create observers subtree
    getObjectManager().create("magenta.GdmoObject*root*observers");

    // create an object to represent server - since it is created
    // in root/observers it will be notified when the queen moves!
    //
    // hostinfo of this AgentProxy should be set by the setServerInfo method,
    // this triggers the Queen::wakeUp method.
    getObjectManager().create("magenta.AgentProxy*root/observers*nqueens-server");
  } // Queen constructor

  /**
   * Queen:setHostInfo -
   *  sets the hostname:port that queen will listen on.
   *  When hostinfo and serverinfo are set then the queen
   *  will "wake up" and begin transmitting its moves.
   *
   * @param newHostInfo   portname:host to listen on
   *
   * @return
   * "success" or "failure reason" if host info is incorrect
   *
   * @version
   * 2001.05.03 alokem creation
   */
  public String
  setHostInfo(String newHostInfo) {
    String retval = super.setHostInfo(newHostInfo);
    
    if ((hostInfoSet == false) && (isSuccess(retval))) {
      hostInfoSet = true;
      wakeUp(hostInfoSet, serverInfoSet);
    }

    return retval;
  } // Queen::setHostInfo

  /**
   * Queen:setServerInfo -
   *  sets the hostname:port of the local nqueens-server proxy.
   *  When hostinfo and serverinfo are set then the queen
   *  will "wake up" and begin transmitting its moves.
   *
   * @param newServerInfo   portname:host of server
   *
   * @return
   * "success" or "failure reason" if server info is incorrect
   *
   * @version
   * 2001.05.03 alokem creation
   */
  public String
  setServerInfo(String newServerInfo) {
    String retval;

    if (serverInfoSet == false) {
      retval = 
       getObjectManager().set("root/observers/nqueens-server*HostInfo*" + newServerInfo);
      if (isSuccess(retval)) {
        serverInfo = newServerInfo;
        serverInfoSet = true;
        wakeUp(hostInfoSet, serverInfoSet);
      }
    } else {
      retval = new String("failure serverinfo change not implemented");
    }

    return retval;
  } // Queen::setServerInfo

  /**
   * Queen::getServerInfo -
   *  return the address of the server this queen is talking to.
   *
   * @return server's address
   * 
   * @version
   * 2001.05.03 alokem creation
   */
  public String
  getServerInfo() {
    return serverInfo;
  } // Queen::getServerInfo

  /**
   * Queen::setPosition -
   *  set row and column where the queen sits on the board.
   *
   * @param positionString    string in form "row,column"
   *
   * @return
   * "success" or "failure reasons"
   *
   * @version
   * 2001.05.04 alokem creation
   */
  public String
  setPosition(String positionString) {
    int newRow, newColumn;

    StringTokenizer tokenizer = new StringTokenizer(positionString, ",");
    if (tokenizer.hasMoreElements()) {
      newRow = Integer.parseInt(tokenizer.nextToken());
      if (newRow > MAX_SIZE) {
        return new String("failure row out of range");
      }
    } else {
      return new String("failure no row specified");
    }

    if (tokenizer.hasMoreElements()) {
      newColumn = Integer.parseInt(tokenizer.nextToken());
      if (newColumn > MAX_SIZE) {
        return new String("failure column out of range");
      }
    } else {
      return new String("failure no column specified");
    }

    row = newRow;
    column = newColumn;
    return new String("success");
  } // Queen::setPosition

  /**
   * Queen::getPosition -
   *  return the row,column where the queen is on the board.
   *
   * @return string in format "row,column"
   *
   * @version
   * 2001.05.04 alokem creation
   */
  public String
  getPosition() {
    return new String(row + "," + column);
  } // Queen:getPosition

  /**
   * Queen::toString -
   *  display information about the Queen including position.
   *
   * @return summary of queen's fields
   *
   * @version
   * 2001.05.04 alokem creation
   */
  public String toString() {
    return new String(super.toString() + " ServerInfo: " + serverInfo +
      " Position: " + row + "," + column);
  } // Queen::toString

  /**
   * Queen::handleEvent -
   *  learns about other queens and reacts to their movements.
   *  More specifically, if we receive an event from an unknown
   *  queen, we add it to *our* list of observers so they will
   *  know what we are doing.
   *  Also, if the latest move of a queen threatens us then
   *  move.
   *
   * @param eventreport   contents of event report.
   *
   * @return 
   * always returns "success"
   * 
   * @version
   * 2001.05.04 alokem creation
   */
  public String
  handleEvent(String eventreport) {
    EventReport event = new EventReport(eventreport);
    ObjectManager om = getObjectManager();

    log("event@" + getHostInfo() + ": " + event);

    // no point in continuing if we can't figure out the following
    if ((event.getHostInfo() == null) ||   // don't know who event is from
        (event.getClassName() == null) ||  // don't know what event is from
        (getHostInfo() == null)) {         // don't know who we are
      return new String("success");
    }

    // if this is an eventreport from another queen
    if ((event.getHostInfo().compareTo(getHostInfo()) != 0) &&
        (event.getClassName().compareTo(this.getClass().getName()) == 0)) {

      // if we don't know about this queen, learn about her
      if (null == om.getObject("root/observers/" + event.getHostInfo())) {
        // add this queen to our observers list
        // create the queen with its name = hostinfo
        om.create("magenta.AgentProxy*root/observers*" + event.getHostInfo());
        om.set("root/observers/" + event.getHostInfo() + "*HostInfo*" +
         event.getHostInfo());
      } // no such queen

      // if this is a move, react to it
      if (event.getAttribute().compareTo("Position") == 0) {
        // if threatened get out of the way
        threatened(event.getValue());
      } // another queen has moved

    } // eventreport is from some other queen

    return new String("success");
  } // Queen::handleEvent

} // class Queen