/*
 * Agent.java
 *
 * history:
 * 2001.04.25 alokem creation
 */

package magenta;

import java.lang.*;
import java.util.*;

/**
 * class Agent -
 *  GdmoObject that forms foundation of Magenta.
 *
 *  <p>
 *  Contains the components:
 *  <ul>
 *  <li><b>ComManager</b> allows the Agent to send and receive messages from 
 *  other Magenta Agents.  
 *
 *  <li><b>ObjectManager</b> is a simple object database.  
 *
 *  <li><b>EventManager</b> manages change notifications from objects 
 *  contained in the ObjectManager and can send change notifications to 
 *  remote agents.  
 *  </ul>
 *
 *  <p>
 *  The Agent class does not handle events or have any interface but it 
 *  contains the foundation upon which these applications can be built.
 *  Managers can be implemented by subclassing Agent and overriding the
 *  handleEvent method and using the hooks into the Agent's subcomponents
 *  provided by getObjectManager, getComManager, etc.
 *
 *  <p>
 *  This class can be invoked from the commandline as follows.
 *  If hostname and port are not specified, they default to localhost:4444.
 *  <pre>
 *  usage: java magenta.Agent [hostname:port] 
 *  </pre>
 *
 *  <dl>
 *  <b>GdmoObject attributes:</b>
 *   <dd>Port (ro) - port number which ComManager will listen on</dd>
 *   <dd>HostName (ro) - hostname for this agent</dd>
 *   <dd>HostInfo - sets port and hostname.  Setting it for the first
          time starts the ComManager.  It cannot be set after
          that.</dd>
 *  </dl>
 *
 * @author aloke mukherjee 
 *
 * @version 
 * 2001.03.11 alokem creation
 */
public class Agent extends AgentProxy {
  /** default host:port that agent will listen on */
  public static String DEFAULTHOSTPORT = "localhost:4444";
  
  /** handle to the internal ComManager component */
  private ComManager cm = null;

  /** handle to the internal EventManager component */
  private EventManager em;

  /** handle to the internal ObjectManager component */
  private ObjectManager om;

  /** hostname where this agent resides */
  private String hostName;

  /** port this agent will listen on */
  private int port;

  /**
   * Agent constructor -
   *  Instantiate the components of a Magenta agent.  This
   *  form of constructor is required since this is derived
   *  class of GdmoObject.
   *
   * @param path    path to the object in the objectmanager
   * @param name    name for this object in the objectmanager
   *
   * @version 
   * 2001.03.11 alokem creation
   */
  public Agent(String path, String name) {
    super(path, name);
    em = new EventManager(this);
    om = new ObjectManager(this);

    // make sure changes in this agent are monitored
    // by its own event manager
    //
    // (note: that since an Agent is a GdmoObject it will
    //        also be event monitored by its creator!)
    this.addObserver(em);
  } // Agent(path, name)

  /**
   * Agent::setHostInfo -
   *  set the hostname:port and start ComManager.
   *  
   * @param newHostInfo   string in format hostname:port
   *
   * @return 
   * success or failure if hostinfo is incorrectly specified.
   *
   * @version 
   * 2001.04.29 alokem creation
   */
  public String
  setHostInfo(String newHostInfo){
    // set newHostInfo to null when the commanager has failed
    if (newHostInfo == null) {
      cm = null;
      super.setHostInfo(null);
      return("success");
    }

    if (cm != null) {
      return new String("failure cannot change hostinfo once " +
       "ComManager is started");
    }

    // expect hostinfo in hostname:port format
   	StringTokenizer	tokenizer = new StringTokenizer(newHostInfo, ":");
    
    if (tokenizer.hasMoreElements()) {
      hostName = tokenizer.nextToken();
    } else {
      return new String("failure no hostname specified");
    }

    if (tokenizer.hasMoreElements()) {
      port = Integer.parseInt(tokenizer.nextToken());
    } else {
      return new String("failure no port specified");
    }

    super.setHostInfo(newHostInfo);

    // create the com manager
    cm = new ComManager(this);
    cm.start();

    return new String("success");
  } // Agent::setHostInfo

  /**
   * Agent::getComManager - 
   *  get handle on ComManager.
   *
   * @return 
   * this Agent's ComManager object
   *
   * @version
   * 2001.03.11 alokem creation
   */
  public ComManager getComManager() {
    return cm;
  } // Agent::getComManager

  /**
   * Agent::getEventManager - 
   *  get handle on EventManager.
   *
   * @return 
   * this Agent's EventManager object
   *
   * @version
   * 2001.03.11 alokem creation
   */
  public EventManager getEventManager() {
    return em;
  } // Agent::getEventManager

  /**
   * Agent::getObjectManager - 
   *  access method for internal ObjectManager component
   *
   * @return 
   * this Agent's ObjectManager object
   *
   * @version
   * 2001.03.11 alokem creation
   */
  public ObjectManager getObjectManager() {
    return om;
  } // Agent::getObjectManager

  /**
   * Agent::log - 
   *  generic output method for debug related messages.  Use this
   *  instead of System.out.println, so that the implementation
   *  of outputting debug messages can be centralized here.  
   *
   *  For example, override this method in a subclass to dump to
   *  message to a file instead of stdout or to provide more
   *  intelligent handling.
   *
   * @param message   String object to be printed out or logged.
   *
   * @version
   * 2001.03.11 alokem creation
   */
  public void 
  log(String message) {
    System.out.println("log: " + message);
  } // Agent::log

  /**
   * Agent::handleEvent -
   *  prints out an eventreport.
   *
   *  Override this method in a subclass to provide more 
   *  intelligent handling.
   *
   * @param eventreport   contents of event report.
   *
   * @return 
   * always returns "success"
   *
   * @version
   * 2001.04.24 alokem creation
   */
  public String 
  handleEvent(String eventreport) {
    System.out.println(getHostInfo() + " event: " + eventreport);
    return new String("success");
  } // Agent::handleEvent

  /**
   * Agent::getPort -
   *  Access function for port member.
   *
   * @return 
   * the port on which this agent will accept new requests.  
   *
   * @version
   * 2001.04.18 alokem creation
   */
  public int
  getPort() {
    return port;
  } // Agent::getPort

  /**
   * Agent::getHostName
   *    Access function for hostName member.
   *
   * @return 
   * the host where this agent resides.
   *
   * @version
   * 2001.04.25 alokem creation
   */
  public String
  getHostName() {
    return hostName;
  } // Agent::getHostName

  /** 
   * Agent main routine -
   *  The entry point for execution of this object.
   *  Parse command-line arguments and instantiate Agent.
   *
   * @param args    the command-line arguments
   *
   * @version 
   * 2001.03.11 alokem creation
   */
  public static void 
  main(String[] args) 	{
    // detects which port the com manager should listen at
    String hostInfo;
    if (args.length < 1) {
      hostInfo = Agent.DEFAULTHOSTPORT;
    } else {
      hostInfo = args[0];
    }

    Agent myAgent = new Agent("", ObjectManager.rootGdmoName);
    myAgent.setHostInfo(hostInfo);
  } // Agent::main
} // class Agent
