package magenta;

import magenta.*;
import java.io.*;
import java.net.*;
import java.util.*;

/**
 * class CLIManagerThread -
 *  The guts of the CLIManager - provides a shell for entering magenta 
 *  commands of the form "destination-agent command".
 *
 * @version
 * 2001.11.10 alokem implement processFile method
 */
public class CLIManagerThread extends Thread {

  /// handle to the CLIManager/Agent which started this thread
  private Agent myAgent;

  /**
   * CLIManagerThread constructor
   *
   * @param theAgent    agent/CLIManager that contains this object
   *
   * @version
   * 2001.11.10 alokem initial comment
   */
  CLIManagerThread(Agent theAgent) {
    super();
    myAgent = theAgent;
  } // CLIManagerThread ctor
  
  /**
   * CLIManagerThread::run -
   *  take care of the input processing thread to allow interactive use.
   *
   * @version
   * 2001.11.10 alokem initial comment
   */
	public void run() {
	  try {
  		this.processInput();
	  } catch (IOException e) {
		  e.printStackTrace();
	  }

	  System.out.println("exiting...");
	  System.exit(0);
	} // CLIManagerThread::run

  /**
   * CLIManagerThread::processInput -
   *  read line from standard input and send to parsing routine.
   *
   * @version
   * 2001.11.10 alokem initial comment
   */
  public void processInput() throws IOException {
	  BufferedReader stdIn = new BufferedReader(new InputStreamReader(System.in));
	  String fromUser;
	  String reply;

	  System.out.print(myAgent.getHostName() + ":" + myAgent.getPort() + "% ");
	  while ((fromUser = stdIn.readLine()) != null) {
      reply = parseQuery(fromUser);
      myAgent.log(reply);
			System.out.print(myAgent.getHostName() + ":" + myAgent.getPort() + "% ");
	  } // while user is entering input
	 
	  stdIn.close();
	} // CLIManagerThread::processInput

  /**
   * CLIManagerThread::parseQuery -
   *  parse and process commands - this is where CLIManager commands
   *  should be handled.  If the query doesn't match any of these it will
   *  be treated as a magenta request and forwarded.
   *
   * @param query   string read from commandline
   *
   * @returns string describing result of operation
   *
   * @version
   * 2001.11.10 alokem moved processing for CLIManager specific commands
   *  (quit, source) here
   * 2001.11.12 alokem allow spaces in queries
   */
	private String 
  parseQuery(String query) throws IOException {
	  StringTokenizer	tokenizer = new StringTokenizer(query, " ");
    String reply = null;
    String firstToken = 
     (tokenizer.hasMoreTokens() ? 
      tokenizer.nextToken() : null);

    // no command was specified
    if (firstToken == null) {
      reply = new String("error: valid commands are quit, source or " +
       "gdmo-name magenta-command");

    // cause program to exit
    } else if (firstToken.compareTo("quit") == 0) {
      System.out.println("exiting...");
      System.exit(1);

    // read in commands from a file
    } else if (firstToken.compareTo("source") == 0) {
      if (tokenizer.hasMoreTokens()) {
        reply = processFile(tokenizer.nextToken());
      } else {
        reply = new String("usage: source filename");
      } // file to source was specified

    // assume this is a valid magenta command
    } else {
      // first token is magenta we're sending this message to
      String agentPath = firstToken;

      if (tokenizer.hasMoreTokens()){
        AgentProxy agent = 
         (AgentProxy)myAgent.getObjectManager().getObject(agentPath);

        if (agent != null) {
          String message = tokenizer.nextToken("\n").trim();
          reply = myAgent.getComManager().send(agent, message);
        } else {
          reply = new String("error: could not find AgentProxy " + agentPath);
        } // if agent was found in local db
      } else {
        reply = new String("error: you did not specify a message to send");
      } // if message was specified
    } // if firsttoken == null
    
    return reply;
	} // CLIManagerThread::parseQuery

  /**
   * CLIManagerThread::processFile -
   *  read lines from a file and treat them as command line commands.
   *
   * @param theFile   string name of file to read from
   *
   * @returns string describing result of operation
   *
   * @version
   * 2001.11.10 alokem creation
   */
  public String
  processFile(String theFile) {
    BufferedReader theReader = null;
    String reply = null;

    if (theFile == null) {
      return new String("error: no file specified to process");
    }

    try {
      theReader = new BufferedReader(new FileReader(theFile));
      String theLine;
      while ((theLine = theReader.readLine()) != null) {
        myAgent.log(parseQuery(theLine));
      } // for each line in file
      theReader.close();
      reply = new String("successfully sourced file " + theFile);
    } catch (FileNotFoundException e) {
      reply = new String("error: couldn't find file to source: " + theFile);
    } catch (IOException f) {
      reply = new String("error: problem sourcing " + theFile);
    }// try to open file reader

    return reply;
  } // CLIManagerThread::processFile

} // class CLIManagerThread