/*
 * EventManager.java
 *
 * history:
 * 2001.04.28 alokem creation
 */

package magenta;

import java.lang.*;
import java.util.*;


/**
 * class EventManager -
 *  monitors all GdmoObjects managed by this Agent.  This class is
 *  automatically registered as an observer of all objects
 *  managed by the ObjectManager.
 *
 *  <p>When an attribute of an object changes, the EventManager's
 *  <code>update</code> method is triggered.  This
 *  will <b>send eventreport messages</b> to all of the objects under the
 *  path <code>root/observers</code> in the object database as well
 *  as the agent which contains it (myAgent).
 *
 * @author aloke mukherjee
 * @version
 * 2001.04.25 alokem creation
 * 2001.07.05 alokem change update routine to reflect change in subTree
 */
public class EventManager implements Observer {
  /** handle on agent to access ObjectManager, ComManager */
  private Agent myAgent;

  /**
   * EventManager constructor -
   *  initialize myAgent member.
   *
   * @param agent   the agent which contains this object
   *
   * @version
   * 2001.04.25 alokem creation
   */
  public EventManager(Agent agent) {
    myAgent = agent;
  } // EventManager constructor

  /**
   * EventManager::update -
   *  send change information to observer agents.  This method is
   *  triggered automatically when the parameter of a GdmoObject 
   *  changes or when a GdmoObject is instantiated or deleted.
   *
   *  <p>EventManager creates a message summarizing the
   *  change.  The message is sent to this agent as well as all
   *  of the agents/proxies in the root/observers subtree.
   *
   *  <p>Event reports contain the following fields separated
   *  by asterisks:
   *  <ul>
   *   <li>eventreport - type of message
   *   <li>hostname:port - agent where change occurred
   *   <li>class - class of changed object
   *   <li>full-gdmo-name - path and name of changed object
   *   <li>attribute - attribute that was changed
   *   <li>value - new value of attribute
   *  </ul>
   *
   *  <p>For example:
   *  <code>
   *  eventreport*boing:4444*Printer*root/Printer0*Instantiated*true
   *  </code>
   *  <br>indicates that a Printer object was created in the
   *  ObjectManager of the Agent running on the host boing and
   *  listening on port 4444.
   *  
   * @param sender   the GdmoObject which triggered the update
   * @param message  string containing name of changed attribute and 
   *                 its new value
   *
   * @version
   * 2001.04.25 alokem creation
   * 2001.07.05 alokem ObjectManager::subTree now includes root node as
   *  first element in result - skip past this
   */
  public void update (Observable sender, Object message) {
    GdmoObject theObject = (GdmoObject)sender;
    ComManager myComManager = myAgent.getComManager();
    ObjectManager om = myAgent.getObjectManager();
    AgentProxy agentProxy;
    SortedMap objectTree;

    // assemble event report
    String eventReport = new String(
      // hostname:port
      myAgent.getHostName() + ":" + myAgent.getPort() + 
        ObjectManager.requestDelimiter +
      // class
      theObject.getClass().getName() +
        ObjectManager.requestDelimiter +
      // full-gdmo-name
      theObject.getPath() + ObjectManager.gdmoNameDelimiter + 
      theObject.getName() + 
        ObjectManager.requestDelimiter +
      // attribute and new value
      message.toString());

    // make sure the agent which contains us knows about the event
    myAgent.handleEvent(eventReport); 
   
    // look up all of the remote agents under root/observers
    // and send notifications to each
    objectTree = om.subTree("root" + ObjectManager.gdmoNameDelimiter + "observers");

    // if no observers then exit
	  if (objectTree == null){
	    return;
	  }

    // send the eventreport to each agentproxy in the list
    Iterator i = (objectTree.values()).iterator();
    // advance past root node (first element in iterator)
    i.next();
    while (i.hasNext()) {
      agentProxy = (AgentProxy)i.next();

      // skip agents whose hostname:port is not set
      if (agentProxy.getHostInfo() != null) {
        try {
          myComManager.sendNoReply(agentProxy, 
              "eventreport" + 
                ObjectManager.requestDelimiter +
              eventReport);
        }	catch (Exception e) { 
          myAgent.log("failed send by EventManager to " + agentProxy.getHostInfo());
        } // end try
      } // hostinfo != null
    } // end while

  } // EventManager::update

} // class EventManager
