/*
 * GdmoObject.java
 *
 * history:
 * 2001.04.25 alokem creation
 */

package magenta;

import java.lang.*;
import java.util.*;
import java.lang.reflect.*;

/**
 * class GdmoObject -
 *  root class for all objects that can be managed by Magenta.
 *  All GdmoObjects have a unique name and a path in the
 *  Magenta's object tree.  
 *
 *  <p>This class includes the <b>set</b> and <b>get</b> methods which
 *  are wrappers for the set/getAttribute methods.  Other magenta
 *  components should use these wrappers to set the attributes
 *  of GdmoObjects since these wrappers take care of looking up
 *  the appropriate method and <b>notifying the EventManager</b> of the
 *  change.
 *
 *  <dl>
 *  <b>GdmoObject attributes:</b>
 *   <dd>Name - unique name for this object in subtree</dd>
 *   <dd>Path - path to this object in ObjectManager</dd>
 *   <dd>Instantiated - set to true when object is created and
 *                      false when it is deleted</dd>
 *  </dl>
 *
 * @author matt ladd, aloke mukherjee
 * @version
 * 2001.02.13 mattladd creation
 * 2001.02.13 alokem factored GdmoObject from Printer, got
 *   reflection working in set() and get() methods
 * 2001.03.05 alokem generate object names using number of times
 *   that ctor has been called
 * 2001.10.22 alokem don't print stack trace when no set method found
 */

public class GdmoObject extends Observable {

  /** name of this object */
  private String name;

  /** path to this object */
  private String path;

  /**
   * set by objectmanager to true after instantiation
   * and to false when the object is deleted - this
   * allows observers to be notified of the object's creation
   * and deletion
   */
  private Boolean instantiated;

  /**
   * keeps track of how many times a certain class has been 
   * instantiated - used to generate unique names.
   * key: class name, value: number of ctor calls
   */
  private static Hashtable classCtorCalls = new Hashtable();

  /**
   * GdmoObject constructor -
   *  set the object's name and path.  All subclasses must declare
   *  this form of constructor and their first line should call this
   *  constructor.
   *
   * <p>If no name is specified, a unique name is generated by 
   * concatenating the className with the number of times that class's 
   * constructor has been called.  That count is maintained in the 
   * classCtorCalls hash.
   *
   * @param path    path to this GdmoObject (defaults to ObjectManager.rootGdmoName)
   * @param name    name of this GdmoObject (defaults to ClassName + instance number)
   *
   * @version
   * 2001.02.13 mattladd creation
   * 2001.02.13 alokem added ctors with different signatures
   * 2001.03.05 alokem generate unique object name
   */
  public GdmoObject(String path, String name) {
    // update number of time's the classes ctor has been called
    Integer id = updateClassCtorCalls();

    if (name == null) {
      // default name uses classname and number of ctor calls
      this.name = (this.getClass()).getName() + id;
    } else {
      this.name = name;
    }

    this.path = ObjectManager.rootGdmoName;
    if (path != null) {
      this.path = path;
    }
  } // GdmoObject constructor

  /**
   * GdmoObject::toString -
   *  display the object's GdmoObject attributes.
   *
   * @return text representation of this object
   *
   * @version
   * 2001.04.06 alokem creation
   */
  public String toString() {
    return new String("class: " + (this.getClass()).getName() + " name: " + name);
  } // GdmoObject::toString

  /**
   * GdmoObject::updateClassCtorCalls -
   *  maintains count of how many times a class has been instantiated.
   *  Used to generate unique object names.
   *
   * @return number of ctor calls minus one
   *
   * @version
   * 2001.03.05 alokem creation
   */
  private Integer updateClassCtorCalls() {
    String className = (this.getClass()).getName();
    Integer id = (Integer)classCtorCalls.get(className);
    if (id == null) {
      id = new Integer(0);
    }
    classCtorCalls.put(className, new Integer(id.intValue() + 1));
    return id;
  } // GdmoObject::updateClassCtorCalls

  /**
   * GdmoObject::getName -
   *  value of this object's name attribute
   *
   * @return this object's name
   *
   * @version
   * 2001.02.13 mattladd creation
   */
  public String getName() {
    return name;
  } // GdmoObject::getName

  /**
   * GdmoObject::getPath -
   *  the value of this object's path attribute
   *
   * @return this object's path
   *
   * @version
   * 2001.03.11 alokem creation
   */
  public String getPath() {
    return path;
  } // GdmoObject::getPath

  /**
   * GdmoObject::setName -
   *  change the value of this object's name attribute.
   *
   * @param name  a new name for this object
   *
   * @return 
   * "success" if successful, otherwise "failure reason"
   *
   * @version
   * 2001.02.13 alokem creation
   */
  public String setName(String name) {
    if (name != null)
    {
      this.name = name;
      return new String("success");
    } else {
      return new String("failure null name");
    }
  }

  /**
   * GdmoObject::setInstantiated -
   *  used to trigger events when object is created or deleted.
   *
   * @param isIinstantiated    "true" if object has just been
   *                           created, "false" if object has just
   *                           been deleted
   *
   * @return
   * "success" if input value was "true" or "false", otherwise "failure"
   *
   * @version
   * 2001.04.25 alokem creation
   */
  public String setInstantiated(String isInstantiated) {
    if (isInstantiated.compareTo("true") == 0) {
      instantiated = Boolean.TRUE;
    } else if (isInstantiated.compareTo("false") == 0) {
      instantiated = Boolean.FALSE;
    } else {
      return new String("failure bad value " + isInstantiated);
    }

    // return "success" to indicate good input
    return new String("success");
  } // GdmoObject::setInstantiated

  /**
   * GdmoObject::set -
   *  wrapper for setAttribute methods that notifies observers.
   *
   * <p>The attribute is specified as a string and reflection is used
   * to get a handle on the appropriate set method.  If the set
   * is successful the object notifies the EventManager.  You <b>must</b>
   * call this method to ensure that the EventManager is notified.
   * 
   * <p>Since the attribute name is used to look up the method, the
   * set method must have a certain signature:
   * <br><code>public String setAttribute(String attrValue)</code>
   *
   * <p>setAttribute should parse attrValue to set the value of an internal 
   * field and return a String beginning with either "success" or "failure"
   * depending on whether the attribute could be set to the specified value.
   *
   * @param attributeName   the attribute to set
   * @param attributeValue  the value to set this attribute to
   *
   * @return
   * string indicating whether set was successful.  
   * If there was no such attribute return value is null.
   *
   * @version
   * 2001.02.14 alokem creation
   * 2001.10.23 alokem don't print stack trace if no set method found
   */
  public String set(String attributeName, String attributeValue) {
    String retval;

    try {
      // set functions are named setAttributeName and take
      // the new attributeValue as an argument of type Object

      // this block of code gets a handle on this method
      String methodName = "set" + attributeName;
      Class params[] = new Class[1];
      params[0] = Class.forName("java.lang.String");
      Class c = this.getClass();
      Method m = c.getMethod(methodName, params);

      // here we create the argument list for the set method and
      // invoke it on ourselves (i.e. this.setAttributeValue(arglist))
      Object arglist[] = new Object[1];
      arglist[0] = attributeValue;
      retval = (String)m.invoke((Object)this, arglist);

      // if first token of retval is "success" notify observers
      StringTokenizer tokenizer = new StringTokenizer(retval, " ");
      if (tokenizer.hasMoreElements() && 
         (tokenizer.nextToken().compareTo("success") == 0)){
        setChanged();
        notifyObservers(attributeName + ObjectManager.requestDelimiter + 
         attributeValue.toString());
      }
    } catch (Exception e) {
      retval = null;
    } // try

    return retval;
  } // GdmoObject::set

  /**
   * GdmoObject::get -
   *  wrapper around a GdmoObject's getAttribute methods.
   *  The attribute is specified as a string and reflection is used
   *  to get a handle on the appropriate get method.  
   * 
   * <p>Since the attribute name is used to look up the method, the
   * get method must have a certain signature:
   * <br><code>public classname getAttribute()</code>
   * <br>where classname is the class of the returned object.  The
   * object should have a toString() method for display.
   *
   * @param attributeName   the attribute to get
   *
   * @return
   * printable object describing the attribute's value or null
   * if there was no such attribute
   *
   * @version
   * 2001.02.14 alokem creation
   */
  public Object get(String attributeName) {
    Object retval;
    String methodName;

   // use reflection on attributeName to call appropriate method
    try {
      // get methods are called getAttributeName and take
      // no arguments

      // this block of code gets a handle on the method
      methodName = "get" + attributeName;
      Class c = this.getClass();
      Method m = c.getMethod(methodName, null);

      // here we invoke the method on ourselves
      retval = m.invoke((Object)this, null);

      // if value is null, return string "(null)" and not
      // the actual value null
      if (retval == null) {
        retval = new String("(null)");
      }
    } catch (Exception e) {
      retval = null;
    } // try  

    return retval;
  } // GdmoObject::get

} // class GdmoObject
