/*
 * ObjectManager.java
 *
 * history:
 * 2001.04.25 alokem creation
 */

package magenta;

import java.lang.*;
import java.util.*;
import java.lang.reflect.*;

/**
 * class ObjectManager -
 *  database of objects managed by the Magenta.
 *  This class manages all of the managed objects using a tree structure.
 *  Each node of the tree has a path and a name which is used to refer
 *  to the object at that node.  The ObjectManager manages this tree.
 * 
 *  <p>The ObjectManager is the only class which knows where 
 *  objects go in the tree.  Most requests specify objects using
 *  their paths.  Because of this <b>all the code to parse create, delete, 
 *  set and get requests is in this class</b> in methods of the same name.
 *
 * @author aloke mukherjee
 * @version
 * 2001.01.23 alokem creation
 * 2001.04.27 alokem added create, delete, set, get, subtree methods
 * 2001.07.04 alokem added simple filtering and scoping to get method,
 *  subTree now returns root node
 * 2001.08.20 alokem debug subTree, add new scope children
 * 2001.08.20 alokem bug fixes in delete, don't allow deletion of root
 * 2001.10.22 alokem change filter syntax to accept multiple attribute/value
 *  pairs
 * 2001.10.27 alokem in createNewObject handle error thrown when classname
 *  is an existing class with incorrect case
 */

public class ObjectManager {

  /** used to delimit gdmo names - root<b>/</b>observers */
  public static final String gdmoNameDelimiter = "/";

  /** used to delimit requests - get*root*HostInfo */
  public static final String requestDelimiter = "*";

  /** used to delimit filter string - attribute=value&attribute=value... */
  public static final String filterDelimiter = "&";

  /** the root gdmo object is "root" */
  public static final String rootGdmoName = "root";

  /** if no class specified create a GdmoObject */
  private static final String defaultClassName = "magenta.GdmoObject";

  /** if no attribute specified for a get, get the object's Name */
  private static final String defaultAttributeName = "Name";

  /** if no filter specified for a get, get all specified objects */
  private static final String defaultFilter = "Name=";

  /** if no scope specified for a get, only get the base object */
  private static final String defaultScope = "baseObject";

  /** the agent which contains us, used to get handles on EventManager, etc. */
  private Agent myAgent;
  
  /** data structure that holds objects */
  private TreeMap objectTree;

  /**
   * ObjectManager constructor -
   *  creates a new object database containing the agent itself as "root".
   *
   * @param a   the agent which contains this objectmanager
   *
   * @version
   * 2001.04.06 alokem creation
   */

  public ObjectManager(Agent a) {
    myAgent = a;

	  objectTree = new TreeMap();
    objectTree.put(rootGdmoName, a);
  } // ObjectManager constructor

  /**
   * ObjectManager::toString -
   *  display contents of the object database.
   *
   * @return
   * string representation of all the objects in the database
   *
   * @version
   * 2001.04.06 alokem creation
   */

  public String toString() {
    StringBuffer buf = new StringBuffer("");

    // iterate through all the objects in the object tree
    Iterator i = (objectTree.keySet()).iterator();
    while (i.hasNext()) {
      String theObjectName = (String)i.next();
      // appends strings of form
      // path/to/object [description of object]
      // description generated by calling the toString method of 
      // each object in the database
      buf.append(theObjectName + 
       " [" + objectTree.get(theObjectName) + "] :: ");
    }

    return buf.toString();
  } // ObjectManager::toString

  /**
   * ObjectManager::create -
   *  parse and handle request to create a new object in database.
   *  In addition to creating the object, this method also will register
   *  the EventManager as an observer of the new object and set the
   *  object's Instantiated field to true.  This will trigger an eventreport
   *  to the agent and any remote observers.
   *
   * @param request   
   * request describing the object to create 
   * <br>syntax: <code>create*classname*parentnode*objectname</code>
   * <br>example: <code>create*Printer*root*Kanata-Printer</code>
   * <ul>
   *  <li>classname - class of object to instantiate - defaults to GdmoObject
   *  <li>parentnode - the container for the new object - defaults to root
   *  <li>objectname - name for the new object - default is auto-generated
   *     (see GdmoObject constructor)
   * </ul>
   *
   * @return 
   * string describing the operation's result
   * <br>syntax: <code>success full-objectname</code> or
   * <code>failure reason</code>
   *
   * @version
   * 2001.04.06 alokem creation
   */
  public String create(String request) {
    StringTokenizer requestTokenizer;
    String className = null;
    String parentGdmoName = null;
    String objectGdmoName = null;
    String fullGdmoName = null;
    GdmoObject newObject;

    if (request != null) {
      requestTokenizer = new StringTokenizer(request, requestDelimiter);
    } else {
      requestTokenizer = new StringTokenizer("", requestDelimiter);
    }

    if (requestTokenizer.hasMoreElements())	{
      className = requestTokenizer.nextToken();
    } else {
      className = defaultClassName;
    }

    if (requestTokenizer.hasMoreElements())	{
      parentGdmoName = requestTokenizer.nextToken();
      // validate that parent object exists in objectTree 
      if (objectTree.get(parentGdmoName) == null) {
        return new String("failure parent node " + parentGdmoName + " does not exist");
      }
    } else {
      parentGdmoName = rootGdmoName;
    }

    // use specified name or use the one generated by the
    // GdmoObject default constructor
    if (requestTokenizer.hasMoreElements()) {
      objectGdmoName = requestTokenizer.nextToken();
    }

    newObject = createNewObject(className, parentGdmoName, objectGdmoName);

    // add object to objectTree
    if (newObject != null)
    {
      // calculate full name for new child
      fullGdmoName = parentGdmoName + gdmoNameDelimiter + newObject.getName();
      objectTree.put(fullGdmoName, (Object)newObject);

      // make sure that other agents know that a new object has been created
      // first, add the EventManager as an observer of this object's attributes
      newObject.addObserver(myAgent.getEventManager());
      // second, set the Instantiated attribute - this will cause EventManager to
      // be notified.  The EventManager will then inform the observing agents.
      newObject.set("Instantiated", "true");
    } else {
      return new String("failure could not create object of type " + className);
    }

    return new String("success " + fullGdmoName);
  } // ObjectManager::create

  /**
   * ObjectManager::createNewObject -
   *  instantiates a new object for inclusion into the object database.
   *  Helper function for create().
   *
   * @param className         class of object to be instantiated
   * @param parentGdmoName    name of the object which will contain
   *  this object.  optional - if null is passed the object will
   *  be created under "root" (see GdmoObject ctor).
   * @param objectGdmoName    name for the object in the database.
   *  optional - if null is passed the object will generate a
   *  name for itself (see GdmoObject ctor).
   *
   * @return
   * the created object or null if the specified
   * class did not exist
   *
   * @version
   * 2001.04.06 alokem creation
   * 2001.10.27 alokem handle NoClassDefFoundError raised when className
   *  is an existing class with incorrect case (i.e. magenta.gdmoobject)
   */
  private GdmoObject createNewObject(String className, String parentGdmoName, String objectGdmoName) {
    // default return value if no such class is available
    GdmoObject theObject = null;

    try {
      // look up ctor for this class that takes parentGdmoName, objectGdmoName
      // as argument
      Class cls = Class.forName(className);
      Class partypes[] = new Class[2];
      partypes[0] = Class.forName("java.lang.String");
      partypes[1] = Class.forName("java.lang.String");
      Constructor ct 
        = cls.getConstructor(partypes);

      // now invoke constructor
      Object arglist[] = new Object[2];
      arglist[0] = (Object)parentGdmoName;
      arglist[1] = (Object)objectGdmoName;
      theObject = (GdmoObject)ct.newInstance(arglist);
    } catch (Exception e) {
      myAgent.log("error: exception while creating a " + className + " - " + e.toString());
    } catch (java.lang.NoClassDefFoundError e) {
      myAgent.log("error: exception while creating a " + className + " - classnames are case sensitive.");
    }

    return theObject;
  } // ObjectManager::createNewObject

  /**
   * ObjectManager::get -
   *  parse and handle a request for an object's attribute.
   *
   * @param request 
   * string specifying the object and attribute to get
   * <br>syntax: <code>get*root-object*attribute-name*filter*scope</code>
   * <br>example: <code>get*root/Kanata-Printer*Status</code>
   * <br>example: <code>get*root*Name*Name=Kanata*wholeSubTree</code>
   * <br>example: <code>get*root*Name*Name=Kan*children</code>
   * <br>example: <code>get*root*Name*Name=Kan&Status=offline*children</code>
   * <ul>
   *  <li>root-object - start searching from this object - defaults to root
   *  <li>attribute-name - name of the attribute to retrieve - defaults to Name
   *  <li>filter - string in the form "attribute=value&attribute=value..." - 
   *   each attribute must exist and have value as a substring to match -
   *   default is to match everything (i.e. no filtering)
   *  <li>scope - baseObject|wholeSubTree|children
   *       <ul>
   *       <li>baseObject (default) - looks only at the root object
   *       <li>wholeSubTree - look at root object and all sub elements
   *       <li>children - look only at subelements - exclude root
   *       </ul>
   * </ul>
   *
   * @return
   * string describing the operation's result
   * <br>syntax: 
   * <code>success fullgdmoname::attribute-name=attribute-value 
   *  fullgdmoname::attribute-name=attribute-value</code>
   * or <code>failure reason</code> - no filter/scope match is a failure
   *
   * @version
   * 2001.04.07 alokem creation
   * 2001.07.04 alokem add basic filtering, scoping capability
   * 2001.08.20 alokem add children scope
   * 2001.10.22 alokem change filter syntax to allow multiple attribute/value
   *  pairs - need to parse and then match all attribute/value pairs
   */
  public String get(String request) {
    StringTokenizer requestTokenizer;
    String fullGdmoName = null;
    String attributeName = null;
    String filter = null;
    String scope = null;
    SortedMap theSubTree = null;
    StringBuffer buf = new StringBuffer("");
    String retval;

    if (request != null) {
      requestTokenizer = new StringTokenizer(request, requestDelimiter);
    } else {
      requestTokenizer = new StringTokenizer("", requestDelimiter);
    }

    // node where search should start from
    if (requestTokenizer.hasMoreElements())	{
      fullGdmoName = requestTokenizer.nextToken();
    } else {
      fullGdmoName = rootGdmoName;
    }

    // check that this object exists in the objectTree
    // and get the subtree starting at the node if it does
    if ((theSubTree = subTree(fullGdmoName)) == null)
      return new String("failure base object not found in database " + fullGdmoName);

    // for matching objects we'll print this attribute
    if (requestTokenizer.hasMoreElements())	{
      attributeName = requestTokenizer.nextToken();
    } else {
      attributeName = defaultAttributeName;
    }

    // search string - only matching objects will be returned
    if (requestTokenizer.hasMoreElements()) {
      filter = requestTokenizer.nextToken();
    } else {
      filter = defaultFilter;
    }

    // parse filter into hash of attribute/value pairs
    StringTokenizer filterTokenizer = 
     new StringTokenizer(filter, filterDelimiter);
    Hashtable filterHash = new Hashtable();
    while (filterTokenizer.hasMoreElements()) {
      String attributeValuePair = filterTokenizer.nextToken();
      StringTokenizer pairTokenizer = 
       new StringTokenizer(attributeValuePair, "=");
      String attribute = pairTokenizer.nextToken();
      String value = 
       (pairTokenizer.hasMoreElements() ? pairTokenizer.nextToken() : "");
      filterHash.put(attribute, value);
    } // for each attribute-value pair in filter
    
    // scope for search - baseObject, wholeSubTree or children
    if (requestTokenizer.hasMoreElements()) {
      scope = requestTokenizer.nextToken();
    } else {
      scope = defaultScope;
    }

    // iterator over the subtree
    Iterator i = (theSubTree.keySet()).iterator();

    // if scope excludes root, advance past it
    if (scope.compareTo("children") == 0) {
      i.next();
    }

    // for each node in the subtree
    //  check that filter string is satisfied
    //  if so print out the requested attribute
    while (i.hasNext()) {
      String theObjectName = (String)i.next();
      GdmoObject theObject = (GdmoObject)objectTree.get(theObjectName);
      Object attributeValue = theObject.get(attributeName);

      // only append to response string if the object had the specified attribute
      // and the attribute value matched the filter
      if ((attributeValue != null) &&
          (doesObjectMatchFilter(theObject, filterHash) == true)) {
        buf.append(theObjectName + "::" + attributeName + "=" + attributeValue + " ");
      }

      // if scope is just the specified object then we are done
      if (scope.compareTo("baseObject") == 0) {
        break;
      }
    }
    
    retval = buf.toString();

    if (retval.compareTo("") != 0) {
      return new String("success " + retval);
    } else {
      return new String("failure no filter matches in scope");
    }
  } // ObjectManager::get

  /**
   * ObjectManager::doesObjectMatchFilter -
   *  match object attributes against filter - helper function for
   *  get().
   *
   * @param theObject   object to match against filter
   * @param filter      hashtable of attribute-value pairs
   *
   * @return true if object matched the filter
   *
   * @version
   * 2001.10.22 alokem creation
   */
  private boolean
  doesObjectMatchFilter(GdmoObject theObject, Hashtable filter) {
    Enumeration attributes = filter.keys();
    boolean retval = true;

    // filter is made up of attribute,value pairs
    // for each attribute in filter - 
    //  if object doesn't have the attribute then no match
    //  if object's value for the attribute doesn't contain the
    //   filter value then no match
    while (attributes.hasMoreElements()) {
      String attribute = (String) attributes.nextElement();
      Object attributeValue = theObject.get(attribute);

      if ((attributeValue == null) || 
          (attributeValue.toString().indexOf((String)filter.get(attribute)) == -1)) {
        retval = false;
        break;
      }
    }

    return retval;
  } // ObjectManager::doesObjectMatchFilter

  /**
   * ObjectManager::set -
   *  parse and handle request to change an object's attribute.
   *
   * @param request
   * string specifying the object and attribute to get
   * <br>syntax: <code>set*fullgdmoname*attribute-name*attribute-value</code>
   * <br>example: <code>set*root/Kanata-Printer*Status*true</code>
   * <ul>
   *  <li>fullgdmoname - complete name of the object - no default
   *  <li>attribute-name - name of the attribute to retrieve - no default
   *  <li>attribute-value - string representation of new attribute value - no default
   * </ul>
   *
   * @return
   * string describing the operation's result
   * <br>syntax: 
   * <code>success fullgdmoname::attribute-name=attribute-value</code>
   * or <code>failure reason</code>
   *
   * @version
   * 2001.04.07 alokem creation
   */
  public String set(String request) {
    StringTokenizer requestTokenizer;
    String fullGdmoName = null;
    String attributeName = null;
    String attributeValue = null;
    GdmoObject theObject = null;
    String retval = null;

    if (request != null) {
      requestTokenizer = new StringTokenizer(request, requestDelimiter);
    } else {
      return new String("failure null request");
    }

    if (requestTokenizer.hasMoreElements())	{
      fullGdmoName = requestTokenizer.nextToken();
    } else {
      return new String("failure no object specified");
    }

    // check that this object exists in the objectTree
    if ((theObject = (GdmoObject)objectTree.get(fullGdmoName)) == null)
      return new String("failure object not found in database " + fullGdmoName);

    if (requestTokenizer.hasMoreElements())	{
      attributeName = requestTokenizer.nextToken();
    } else {
      return new String("failure no attribute specified");
    }

    if (requestTokenizer.hasMoreElements()) {
      attributeValue = requestTokenizer.nextToken();
    } else {
      return new String("failure no attribute value specified");
    }

    retval = theObject.set(attributeName, attributeValue);
    
    if (retval == null) {
      return new String("failure " + fullGdmoName + " does not have attribute " + attributeName);
    } else {
      return retval;
    }
  } // ObjectManager::set

  /**
   * ObjectManager::delete -
   *  parse and handle requests to delete objects and their children.
   *  This method set's the object's Instantiated field to false which
   *  triggers an eventreport to the agent and any remote observers.
   *
   * @param request
   * string specifying the node to delete
   * <br>syntax: <code>delete*fullgdmoname</code>
   * <br>example: <code>delete*root/Kanata-Printer</code>
   * <ul>
   *  <li>fullgdmoname - complete name of the object - no default
   * </ul>
   *
   * @return
   * string indicating success of operation
   * <br>syntax: <code>success fullgdmoname</code> or
   * <code>failure reason</code>
   *
   * @version
   * 2001.04.07 alokem creation
   * 2001.08.20 alokem notify for deleted subnodes, prevent root deletion
   */
  public String delete(String request) {
    StringTokenizer requestTokenizer;
    String fullGdmoName = null;
    GdmoObject theObject = null;

    if (request != null) {
      requestTokenizer = new StringTokenizer(request, requestDelimiter);
    } else {
      return new String("failure null request");
    }

    if (requestTokenizer.hasMoreElements())	{
      fullGdmoName = requestTokenizer.nextToken();
    } else {
      return new String("failure no object specified");
    }

    // prevent deletion of root
    if (fullGdmoName.compareTo(rootGdmoName) == 0) {
      return new String("failure cannot delete root");
    }

    // check that this object exists in the objectTree
    if ((theObject = (GdmoObject)objectTree.get(fullGdmoName)) == null)
      return new String("failure object not found in database " + fullGdmoName);

    Iterator keyIterator = (objectTree.tailMap(fullGdmoName)).keySet().iterator();
    while (keyIterator.hasNext()) {
      String theKey = (String)keyIterator.next();

      // matches the node to delete (root) and all of its subnodes
      if (theKey.indexOf(fullGdmoName) == 0) { 
        theObject = (GdmoObject)objectTree.get(theKey);
        // make sure EventManager is notified about this deletion
        theObject.set("Instantiated", "false");
        keyIterator.remove(); 
      } else {
        // root node is not prefix to this node, we are in a new subtree 
        // so quit
        break;
      } // if this is node to delete or subnode
    } // for each node starting at node to delete

    return new String("success " + fullGdmoName);
  } // ObjectManager::delete

  /**
   * ObjectManager::subTree -
   *  return the collection consisting of node and its children objects.
   *
   * @param low   root node of the subtree, for example "root/observers"
   *
   * @return collection consisting of node and its children objects.  the 
   *         node will be the first element
   *
   * @version
   * 2001.04.25 alokem creation
   * 2001.07.04 alokem change to include root node
   * 2001.08.20 alokem fix bug where leaf node subTrees were returning null
   */
  public SortedMap subTree(String low) {
    String high = null;

    // make sure this node exists
    if (objectTree.get(low) == null) {
      return null;
    }
    
    // traverse subMap until first key that doesn't have prefix of gdmoPath
    Iterator i = (objectTree.tailMap(low)).keySet().iterator();
    while (i.hasNext()) {
      high = (String)i.next();
      if (high.indexOf(low) != 0)
        break;
    }

    // this is weird:
    // if high and low are the same value then we return the trivial
    // tree consisting of the specified value.
    //
    // if the last element of the subtree is also the last element
    // in the objectTree then high will be the last element - include it
    // in the result set by adding "\0".
    //
    // if the last element of the subtree is NOT the last element
    // in the objectTree then high will be first element past the
    // end of the subTree, so don't include it (i.e. don't add
    // "\0").
    //  
    // See behaviour of submap for more info.
    if (high.compareTo(low) == 0) {
      high = low + "\0";
    } else if (high.indexOf(low) == 0) {
      high = high + "\0";
    }
    return objectTree.subMap(low, high);
  } // ObjectManager::subTree

  /**
   * ObjectManager::getObject -
   *  retrieve an object from the database using it's full gdmo name.
   *
   * @param fullGdmoName  string identifying path in database, for example
   *                      root/Kanata-Printer.
   *
   * @return
   * the specified object if found or null
   *
   * @version
   * 2001.04.25 alokem creation
   */
  public GdmoObject 
  getObject(String fullGdmoName) {
    return (GdmoObject)objectTree.get(fullGdmoName);
  } // ObjectManager::getObject
} // class ObjectManager
