
/**
  * Implements a swap manager, which coordinates the transition of replacing
  * an old bean with a new bean. The old bean and the new bean are not necessarily
  * implements the same interface.
  * @author Lei Tan
  */
package carleton.swapbox;

import java.io.*;
import java.util.*;
import java.lang.reflect.Method;
import java.beans.*;
import java.lang.reflect.InvocationTargetException;
import org.xml.sax.*;
import org.w3c.dom.*; 
import javax.xml.parsers.*;
import sun.beanbox.*;

public class SwapManager implements Serializable, IsIdleListener, TimerRequester{
    /**
      * Wrapper is an existed class in beanbox package. It provides many
      * facilities to manipulate behaviors at beans, e.g., add/remove
      * listeners, displaying focus box and so on
      */
    private Wrapper newWrapper;
    private Wrapper oldWrapper;
    
    /** bb is used when the old bean is being purged off from displaying
      * beanbox. It has nothing to do with the swap transaction itself.
      */
    private SwapBox bb;
    private boolean expire;
    private SwapConfiguration config;
    
    /**
      * Construct a SwapManager
      */
    public SwapManager (Wrapper newWrapper, Wrapper oldWrapper, SwapBox bb) {
    //public SwapManager (Wrapper newWrapper, Wrapper oldWrapper) {
        this.newWrapper = newWrapper;
        this.oldWrapper = oldWrapper;
        this.bb = bb; 
    }
    
    /** Swap the old bean with the new bean. This is a no state-transfer swap*/    
    public void swap_Stateless() throws SwapException {
        Object oldBean = oldWrapper.getBean();
        Object newBean = newWrapper.getBean();
        
        ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(false);
        createListenersForNewBean();
        blockOldBeanService();
        
        Class c = oldBean.getClass();
        Method[] methods = c.getMethods();
        boolean found = false;
        for (int i = 0; i < methods.length; i++) {
            if (methods[i].getName().equals("addIsIdleListener")) {
                try {
                    Object args[] = {this};
                    methods[i].invoke(oldBean, args);
	            } catch (Exception ex) {
	                unblockService(oldWrapper);
	                cleanup(newWrapper);
	                ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
	                throw new SwapException("SwapManager: adding event listener for " + 
	                                        methods[i].getName() + " failed" + "\n   " + ex.getMessage());
	            }
	            found = true;
	            break;
	        }
	    }
	    
	    /** If notFound is still true, there is no addIsIdleListener method in the old S-Module.
	      * The old S-Module should be deleted from the BeanBox right now rather than later on
	      * after it finishes the pending task
	      */
	 	if (!found) {
	 	    unblockService(newWrapper);
	        cleanup(oldWrapper);
	        ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
	    }
    }
    
    public void targetModuleIsIdle() {
        /**
          * All listeners must be removed from the old bean so that it can be garbege collected
          * by the JVM later on. All listeners except IsIdleListener is added at old bean's wrapper, therefore
          * can be removed by removeListener method at Wrapper. The following code first remove 
          * IsIdleListener by invoking the remover via reflection, then remove all other listeners by
          * invoking removeListener method at Wrapper
          */
		Object oldBean = oldWrapper.getBean();
		Class c = oldBean.getClass();
        Method[] methods = c.getMethods();
        boolean found = false;
        
        for (int i = 0; i < methods.length; i++) {
            if (methods[i].getName().equalsIgnoreCase("removeIsIdleListener")) {
                try {
                    Object args[] = {this};
                    methods[i].invoke(oldBean, args);
	            } catch (Exception ex) {
	                unblockService(oldWrapper);
	                cleanup(newWrapper);
	                ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
	            }
	            found = true;
	            break;
	        }
	    }
	    
	    if (!found) {
	        unblockService(oldWrapper);
	        cleanup(newWrapper);
	        ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
	        return;
	    }
	    
	    unblockService(newWrapper);
	    cleanup(oldWrapper);
	    ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
    }
    
    public void swap_Stateful (String xmlDName, String xmlFName) throws SwapException{
        Object newBean = newWrapper.getBean();
        Object oldBean = oldWrapper.getBean();
        
        ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(false);
        //System.out.println("I am trying to parse the configuration file");
        try {
            config = parseConfigurationXML(new File(xmlDName, xmlFName));
        } catch (SwapException se) {
            ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
            throw se;
        }
        //System.out.println("I am finished parsing the configuration file");
        
        expire = false;
        (new Timer(this)).request(config.getTimeConstraint(), false);
        createListenersForNewBean();
        blockOldBeanService();
            
        synchronized (this) {
            if (expire) {
                unblockService(oldWrapper);
                cleanup(newWrapper);
                ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
                throw new SwapException("SwapManager: Time up. Swap Transaction aborted");
            }
        }
        
        try {
            //createStatesForNewBean(xmlDName, xmlFName);
            createStatesForNewBean();
        } catch (SwapException se) {
            unblockService(oldWrapper);
            cleanup(newWrapper);
            ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
            throw se;
        }
        
        unblockService(newWrapper);
        Method[] newMethods = newBean.getClass().getDeclaredMethods();
        boolean found = false;
        for (int i = 0; i < newMethods.length; i++) {
            if (newMethods[i].getName().equalsIgnoreCase("swapMethod")) {
                try {
                    Object args[] = {};
                    newMethods[i].invoke(newBean, args);
                } catch (Exception ex) {
                    unblockService(oldWrapper);
                    cleanup(newWrapper);
                    ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(true);
                    throw new SwapException("SwapManager: Invoke method at new Module failed" +
                                            "\n   " + ex.getMessage());
                }
                found = true;
                break;
            }
        }
        
        if (!found) {
            throw new SwapException("SwapManager: There is no swapMethod() at the new S-Module");
        }
        cleanup(oldWrapper);
        ((SwapBox)BeanBoxFrame.getCurrentBeanBox()).setEventsMenu(false);
    }     
    
    public synchronized void timeUp() {
        expire = true;
    }
    
    /**
      * Return true if class a is either equivalent to class b, or if
      * class a is a subclass of class b.
      * Note that either or both "Class" object may represent interfaces.
      * @param a the class to be compared
      * @param b the benchmark class
      */
    private static boolean isSubclass (Class a, Class b) {
        /**
          * We rely on the fact that for any given java class or primitive type
          * there is a unique Class object, so we can use object equivalence in
          * the comparisons.
          */
        if (a == b) {
            return true;
        }
        if (a == null || b == null) {
            return false;
        }
        for (Class x = a; x != null; x = x.getSuperclass()) {
            if (x == b) {
                return true;
            }
            if (b.isInterface()) {
                Class interfaces[] = x.getInterfaces();
                for (int i = 0; i < interfaces.length; i++) {
                    if (interfaces[i] == b) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
    
    private class SwapXMLHelper extends HandlerBase {
        private Vector newNames, oldNames;
        
        public SwapXMLHelper (Vector newNames, Vector oldNames) {
            this.newNames = newNames;
            this.oldNames = oldNames;
        }
        
        public void startElement (String name, AttributeList attrs) throws SAXException {
            if (name.equalsIgnoreCase("state")) {
                newNames.addElement(attrs.getValue(0));
                oldNames.addElement(attrs.getValue(1));
            } 
        }
    }
    
    private void createListenersForNewBean() {
        Object oldBean = oldWrapper.getBean();
        Object newBean = newWrapper.getBean();
        SwapReport aReport = AdapterCenter.report(newBean, oldBean);
        Vector v = aReport.getUnchangedEvents(); // array may be better because it is type safe
        for (int i = 0 ; i < v.size(); i++){
            SwapEventInfo sei = (SwapEventInfo)v.elementAt(i);
            // In  order to achieve block effect, we block the service of the old bean here rather 
            // than later after creating adapters for the new bean. This can demonstrate our 
            // block strategy does work. In the real case, it is better to block the service of the
            // old bean after creating the adapters for the new bean, because it is a time-consuming
            // job and swap transaction is time sensitive.
            // <p>
            // Only if the adapter is the target does it need to block the service.
            if (sei.isTarget(oldBean)) {
                SwapHookupManager.hookup(sei.getEvent(), sei.getListenerMethod(), sei.getSource(),
                                     newWrapper, sei.getTargetMethod(), sei.getEventName(), true);
                AdapterCenter.addInvalidWrapper(newWrapper, oldWrapper);

			} else {
				SwapHookupManager.hookup(sei.getEvent(), sei.getListenerMethod(), newWrapper,
				                     sei.getTarget(), sei.getTargetMethod(), sei.getEventName(), true);
				AdapterCenter.addInvalidWrapper(newWrapper, oldWrapper);
			}
        }
    }              
    
    private void blockOldBeanService() {
        BeanReport br = AdapterCenter.getBeanReport(oldWrapper.getBean());
        Vector oldBeanIncoming = br.getIncomingEvents();
        
        br = AdapterCenter.getBeanReport(newWrapper.getBean());
        Vector newBeanIncoming = br.getIncomingEvents();
        Vector newBeanOutgoing = br.getOutgoingEvents();
        
        // block incoming events for the old bean. Those events are queued at the adapter.
        for (int i = 0; i < oldBeanIncoming.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)oldBeanIncoming.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setBlock(true);
        }
        
        // set new bean able to accept incoming events. Those evetns are queued at the adapter
        for (int i = 0; i < newBeanIncoming.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)newBeanIncoming.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setService(true);
        }
        for (int i = 0; i < newBeanOutgoing.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)newBeanOutgoing.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setService(true);
        }
    }    
    
    private void unblockService(Wrapper aWrapper) {
        BeanReport br = AdapterCenter.getBeanReport(aWrapper.getBean());
        Vector incomingEvents = br.getIncomingEvents();
        for (int i = 0; i < incomingEvents.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)incomingEvents.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setBlock(false);
        }
        if (aWrapper == newWrapper) {
            Vector outgoingEvents = br.getOutgoingEvents();
            for (int i = 0; i < outgoingEvents.size(); i++) {
                SwapEventInfo sei = (SwapEventInfo)outgoingEvents.elementAt(i);
                ((SwapAdapter)sei.getAdapter()).setBlock(false);
            }
        }
    }
            
    
    private SwapConfiguration parseConfigurationXML (File xmlFile) throws SwapException {
        Vector newNames = new Vector();
        Vector oldNames = new Vector();
        long timeConstraint = -1;
        Document document;
        
        DocumentBuilderFactory factory =
           DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            document = builder.parse(xmlFile);

        } catch (SAXParseException spe) {
            throw new SwapException("SwapManager: Parsing error, line " + spe.getLineNumber () +
                                    ", uri " + spe.getSystemId () + "   " + spe.getMessage());
        } catch (SAXException sxe) {
           throw new SwapException("SwapManager: An error occured when initialize XML parser" +
                                    "\n   " + sxe.getMessage());
        } catch (ParserConfigurationException pce) {
            throw new SwapException("SwapManager: Initialize new parser failed" + 
                                    "\n   " + pce.getMessage());
        } catch (IOException ioe) {
            throw new SwapException("SwapManager: I/O failed" + "\n   " + ioe.getMessage());
        }
        
        NodeList nodeList = document.getElementsByTagName("state");
        Node node;
        for (int i = 0; i < nodeList.getLength(); i++) {
            node = nodeList.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                Element element = (Element)node;
                newNames.addElement(element.getAttribute("newName"));
                oldNames.addElement(element.getAttribute("oldName"));
            } else {
                throw new SwapException("SwapManager: Parsing XML file failed");
            }
        }
        if (newNames.size() != oldNames.size()) {
            throw new SwapException("SwapManager: State Transfer not in Pair");
        }
        /*for (int i = 0; i < newNames.size(); i++) {
            System.out.println("\nThe old name is: " + (String)oldNames.elementAt(i));
            System.out.println("The new name is: " + (String)newNames.elementAt(i));
        }*/
        
        
        nodeList = document.getElementsByTagName("time");

        if (nodeList.getLength() == 1 && nodeList.item(0).getNodeType() == Node.ELEMENT_NODE) {
            NodeList textNode = nodeList.item(0).getChildNodes();
            if (textNode.getLength() == 1 && textNode.item(0).getNodeType() == Node.TEXT_NODE
                && textNode.item(0).getNodeValue() != null) {
                    
                try {
                    timeConstraint = (new Long(textNode.item(0).getNodeValue())).longValue();
                } catch (NumberFormatException nfe) {
                    throw new SwapException("SwapManager: Error when converting string to long" +
                                            "\n    The string is not in number format");
                }
            }
        } else {
            throw new SwapException("SwapManager: Parsing timeConstraint failed");
        }     
        
        if (timeConstraint < 0) {
            throw new SwapException("SwapManager: TimeConstraint is wrong " +
                                    "\n    The value is: " + timeConstraint);
        }
        
        return new SwapConfiguration(newNames, oldNames, timeConstraint);
    }
    
    //private void createStatesForNewBean (String xmlDName, String xmlFName) throws SwapException {
    private void createStatesForNewBean () throws SwapException {
        Object newBean = newWrapper.getBean();
        Object oldBean = oldWrapper.getBean();
        Vector newNames;
        Vector oldNames;
        
        if (config == null) {
            throw new SwapException("SwapManger: The config file is null");
        } else {
            newNames = config.getNewNames();
            oldNames = config.getOldNames();
            if (newNames == null || oldNames == null) {
                throw new SwapException("SwapManager: The XML has semantical error");
            }
        }
     
        /*for (int i = 0; i< newNames.size(); i++) {
            System.out.println("\nThe old name is: " + (String)oldNames.elementAt(i));
            System.out.println("The new name is: " + (String)newNames.elementAt(i));
        }*/
        
        /*long timeConstraint = -1;
        Document document;
        
        DocumentBuilderFactory factory =
           DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            document = builder.parse( new File(argv[0]) );

        } catch (SAXParseException spe) {
            throw new SwapException("SwapManager: Parsing error, line " + spe.getLineNumber () +
                                    ", uri " + spe.getSystemId () + "   " + spe.getMessage());
        } catch (SAXException sxe) {
           throw new SwapException("SwapManager: An error occured when initialize XML parser" +
                                    "\n   " + sxe.getMessage());
        } catch (ParserConfigurationException pce) {
            throw new SwapException("SwapManager: Initialize new parser failed" + 
                                    "\n   " + pce.getMessage());
        } catch (IOException ioe) {
            throw new SwapException("SwapManager: I/O failed" + "\n   " + ioe.getMessage());
        }
        
        NodeList = document.getElementsByTagName("state");*/
        
  
        /*SAXParserFactory factory;
        SAXParser saxParser;
        try {
            factory = SAXParserFactory.newInstance();
            saxParser = factory.newSAXParser();
        } catch (ParserConfigurationException ex) {
            throw new SwapException("SwapManager: Initialize new parser failed" + 
                                    "\n   " + ex.getMessage());
        } catch (SAXException ex) {
            throw new SwapException("SwapManager: An error occured when initialize XML parser" +
                                    "\n   " + ex.getMessage());
        }
        
  
        try {
            saxParser.parse(new File(xmlDName, xmlFName), new SwapXMLHelper(newNames, oldNames));
        } catch (Exception ex) {
            throw new SwapException("SwapManager: Parse XML file failed. The swap transaction stops" +
                                    "\n   " + ex.getMessage());
        }*/
        
        /*if (newNames.size() != oldNames.size()) {
            throw new SwapException("SwapManager: XML file has sementical error" + 
                                    "\n   The state name attribute is not in pair");
        }*/
        
        Class oldClass = oldBean.getClass();
        Class newClass = newBean.getClass();
        Method[] oldMethods = oldClass.getDeclaredMethods();
        Method[] newMethods = newClass.getDeclaredMethods();
        Method oldGetter = null;
        Method newSetter = null; 
        String newName = null;
        String oldName = null; 
        
        for (int i = 0; i < newNames.size(); i++) {
            oldName = (String)oldNames.elementAt(i);
            for (int j = 0; j < oldMethods.length; j++) {
                if (oldMethods[j].getName().equalsIgnoreCase("swapGet" + oldName)||
                    oldMethods[j].getName().equalsIgnoreCase("get" + oldName)||
                    oldMethods[j].getName().equalsIgnoreCase("is" + oldName)) {
                        oldGetter = oldMethods[j];
                        //System.out.println("The old getter is: " + oldGetter);
                        break;
                    }
            }
            
            newName = (String)newNames.elementAt(i);
            for (int j = 0; j < newMethods.length; j++) {
                if (newMethods[j].getName().equalsIgnoreCase("swapSet" + newName)||
                    newMethods[j].getName().equalsIgnoreCase("set" + newName)){
                    newSetter = newMethods[j];
                    //System.out.println("The new setter is: " + newSetter);
                    break;
                }
            }
            
            if (oldGetter == null) {
                throw new SwapException("SwapManager: There is no getter method for" +
                                        "\nstate named " + oldName + " at the old S-Module");
            }
            if (newSetter == null) {
                throw new SwapException("SwapManager: There is no setter method for" +
                                        "\nstate named " + newName + " at the new S-Module");
            }
            
            try {
                Object oldArgs[] = {};
                Object newArgs[] = {oldGetter.invoke(oldBean, oldArgs)};
                newSetter.invoke(newBean, newArgs);
            } catch (Exception ex) {
                ex.printStackTrace();
                throw new SwapException("SwapManager: Transfering state failed" +
                                        "\n   " + ex.getMessage());
	        }
	        
	        oldGetter = null;
	        newSetter = null;
	        oldName   = null;
	        newName   = null;
        }
    }
    
    private void cleanup(Wrapper aWrapper) {
        /**
          * Since the Wrapper class does not store information of events in which 
          * it is the target. We have to find it out at AdapterCenter class, and
          * then clean them up at the source wrapper.
          */
        BeanReport br = AdapterCenter.getBeanReport(aWrapper.getBean());
        if (br.getIncomingEvents().size() > 0) {
            Vector v = br.getIncomingEvents();
            Vector tempV = (Vector)v.clone();
            for (int i = 0; i < tempV.size(); i++) {
                SwapEventInfo sei = (SwapEventInfo)tempV.elementAt(i);
                Wrapper source = sei.getSource();
                source.swapRemoveOneListener(sei);
            }
        }

        /**
          * Now we remove adaptors in which aWrapper is the source. 
          */
        if (aWrapper == oldWrapper) {
            aWrapper.swapRemoveListeners();
            AdapterCenter.removeOne(aWrapper, true);
            bb.removeOne(aWrapper);
            //BeanBoxFrame.getCurrentBeanBox().removeOne(aWrapper);
        } else if(aWrapper == newWrapper) {
            aWrapper.swapRemoveListeners();
            aWrapper.listenForMice(true);
            AdapterCenter.removeOne(aWrapper, false);
        }
                       
        /**
          * It's time to unblock services at adaptors
          */
        if (aWrapper == oldWrapper) {
            br = AdapterCenter.getBeanReport(newWrapper.getBean());
        } else if (aWrapper == newWrapper) {
            br = AdapterCenter.getBeanReport(oldWrapper.getBean());
        }
        Vector incomingEvents = br.getIncomingEvents();
        Vector outgoingEvents = br.getOutgoingEvents();
        for (int i = 0; i < incomingEvents.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)incomingEvents.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setService(true);
            ((SwapAdapter)sei.getAdapter()).setBlock(false);
        }
        for (int i = 0; i < outgoingEvents.size(); i++) {
            SwapEventInfo sei = (SwapEventInfo)outgoingEvents.elementAt(i);
            ((SwapAdapter)sei.getAdapter()).setBlock(true);
            ((SwapAdapter)sei.getAdapter()).setBlock(false);
        }
    }
}


    