/** Agent.java in the package org.JIFSA of the JIFSA project.
	Originally created 17-Jun-07 
    
    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */


package org.JIFSA;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.preprocessing.filters.CaseFilter;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;

/** The Agent class is used to create an imitative software agent.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class Agent {

	//the case base used by the agent
	private CaseBase m_caseBase;
	//used to find the closest cases in the case base
	private CaseBaseSearch m_cbSearch;
	//used to determine the action to perform, based on closest cases
	private ActionSelection m_actionSelection;
	//the filters that are applied to each new Case
	private CaseFilter[] m_filters;
	
	/** The constructor to create an Agent object.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public Agent(CaseBase cb, CaseBaseSearch cbs, ActionSelection as){
		//check parameters
		if(cb == null){
			throw new IllegalArgumentException("Null CaseBase given to Agent.");
		}
		if(cb.getCasebaseSize() == 0){
			throw new IllegalArgumentException("Empty CaseBase given to Agent. At least one Case must be in CaseBase.");
		}
		if(cbs == null){
			throw new IllegalArgumentException("Null CaseBaseSearch given to Agent.");
		}
		if(as == null){
			throw new IllegalArgumentException("Null ActionSelection given to Agent.");
		}
		
		this.m_caseBase = cb;
		this.m_cbSearch = cbs;
		this.m_actionSelection = as;
		this.m_filters = new CaseFilter[0];
	}
	
	/** This method is used to allow the agent to sense and react to
	 * it's environment. Given the agent's current inputs
	 * the agent will perform an action. A null return value means
	 * no appropriate action could be found.
	 * 
	 * @param sc The agent's current inputs
	 * @return The action performed by the agent
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public AgentAction senseEnvironment(AgentInputs av){
		//check for invalid input
		if(av == null){
			throw new IllegalArgumentException("Null AgentInputs given to Agent.");
		}
		
		//first make it into a Case with no actions
		Case currentVision = new Case(av, new ArrayList<AgentAction>());
		
		//apply any filters that have been set
		for(int ii=0; ii < this.m_filters.length; ii++){
			currentVision = this.m_filters[ii].filter(currentVision);
		}
		
		//get the closest cases in the casebase
		List<Case> closest = this.m_cbSearch.findClosest(currentVision,this.m_caseBase);
		return this.m_actionSelection.selectAction(closest);
	}
	
	/** Sets the filters that will be applied to each new Case the Agent
	 * encounters in the senseEnvironment method.
	 * 
	 * @param filters The filters to use
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public void setFilters(CaseFilter[] filters){
		for(int ii=0;ii<filters.length;ii++){
			if(filters[ii] == null){
				throw new IllegalArgumentException("Null CaseFilter given to agent.");
			}
		}
		
		this.m_filters = filters;
	}
	
	
	/** Used to change the CaseBase used by the Agent
	 * 
	 * @param cb the new CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public void setCaseBase(CaseBase cb){
		if(cb == null){
			throw new IllegalArgumentException("Null CaseBase given to Agent.");
		}
		if(cb.getCasebaseSize() == 0){
			throw new IllegalArgumentException("Empty CaseBase given to Agent. At least one Case must be in CaseBase.");
		}
		
		this.m_caseBase = cb;
		
	}
	

}
