/** AgentInputs.java in the package org.JIFSA of the JIFSA project.
    Originally created 18-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    * 
    */

package org.JIFSA;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;


/** Contains the objects currently sensed by the agent
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class AgentInputs implements Serializable {


    private static final long serialVersionUID = -6016212103321562592L;
	
    /** data collection the other collections */
    private Hashtable<String,List<SensoryItem>> m_featuregroupTable;

	
    /** Default constructor creates an empty AgentInputs object
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public AgentInputs(){
	//The hashtable seperates the features based on their type, for easy 
	//access to the different types
	this.m_featuregroupTable = new Hashtable<String,List<SensoryItem>>();
    }
	
    /** Copy constructor
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public AgentInputs(AgentInputs v){
	//test params
	if(v == null){
	    throw new IllegalArgumentException("Null AgentVision given to copy constructor.");
	}
		
	//set the global variables
	this.m_featuregroupTable = new Hashtable<String,List<SensoryItem>>(v.m_featuregroupTable);
		
    }

    public Hashtable<String,List<SensoryItem>> getSensoryItems(){
	return new Hashtable<String,List<SensoryItem>>(this.m_featuregroupTable);
    }

    public void setSensoryItems(Hashtable<String,List<SensoryItem>> items){
	this.m_featuregroupTable=new Hashtable<String,List<SensoryItem>>(items);
    }
	
    /** Adds a feature to the list of features that exist in the 
     * AgentInputs.
     * 
     * @param f The feature to be added
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void addSensoryItem(SensoryItem f){
	if(f == null){
	    throw new IllegalArgumentException("Null SensoryItem added to AgentVision.");
	}
		
	//check to see if the feature type has a key in the hashtable
	if(this.m_featuregroupTable.containsKey(f.getFeatureName())){
	    //if this feature has a key, then it already has a list
	    //  we can append it to
	    List<SensoryItem> l = this.m_featuregroupTable.get(f.getFeatureName());
	    l.add(f);
	    //update the value in the hashtable
	    this.m_featuregroupTable.put(f.getFeatureName(), l);
	}else{
	    //create a new List and add to it
	    List<SensoryItem> l = new ArrayList<SensoryItem>();
	    l.add(f);
	    //update the value in the hashtable
	    this.m_featuregroupTable.put(f.getFeatureName(), l);
	}
		
    }
	
    /** Checks to see if any features of a specific type exist
     * in the AgentInputs object.
     * 
     * @param featureName The name of the feature type of check for
     * 
     * @return true if at least one feature of that type is in the AgentInputs
     * 
     * @author Michael W. Floyd
     * @since 0.1
     * 
     */
    public boolean doesContain(String featureName){
	//check parameters
	if(featureName == null){
	    throw new IllegalArgumentException("SensoryItem name was null.");
	}
		
	//get the list of that type of feature
	List<SensoryItem> l = this.m_featuregroupTable.get(featureName);
		
	//if there is no entry in the feature table the HashMap will
	//have returned null
	if(l == null || l.size() == 0){
	    return false;
	}
	return true;
    }
	
    /** Returns the number of features in the AgentInputs
     * of a particular type.
     * 
     * @param featureName The name of the feature type
     * 
     * @return The number of features in the AgentInputs of a given type
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public int getNumberSensoryItems(String featureName){
	//check parameters
	if(featureName == null){
	    throw new IllegalArgumentException("SensoryItem name was null.");
	}
		
	//get the list for that feature
	List<SensoryItem> sensoryItemList = this.m_featuregroupTable.get(featureName);
	//if null then no features of that type exist
	if(sensoryItemList == null){
	    return 0;
	}
	return sensoryItemList.size();
    }
	
    /** Returns the total number of features in the AgentInputs
     * 
     * @return The number of features in the AgentInputs
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public int getNumberSensoryItems(){
	//get the feature list for each type of feature
	Collection<List<SensoryItem>> lists = this.m_featuregroupTable.values();
		
	int featureCounter = 0;
		
	for(List<SensoryItem> currentList : lists){
	    //ignore null lists, these occur when we remove a feature list
	    if(currentList != null){
		featureCounter += currentList.size();
	    }
	}
		
	return featureCounter;
    }
	
    /** Removes all features of a specific type from the AgentVision
     * 
     * @param featureName The feature type to remove
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void removeSensoryItemType(String featureName){
	//check parameters
	if(featureName == null){
	    throw new IllegalArgumentException("SensoryItem name was null.");
	}
		
	//get the list for the specified feature
	List<SensoryItem> sensoryItemList = this.m_featuregroupTable.get(featureName);
		
	//if it is null then the feature doesn't exist in the AgentVision
	if(sensoryItemList != null){
	    //clear the list
	    sensoryItemList.clear();
	    //change the table entry to null  (meaning empty)
	    this.m_featuregroupTable.remove(featureName);
	}
    }
	
    /** Returns a list of all features of a specific type. If no
     * features of that type exist, a null value will be returned.
     * 
     * @param featureName The name of the feature type to return
     * @return A list of features (or null if the feature does not exist in the AgentInputs)
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public List<SensoryItem> getSensoryItems(String featureName){
	//check parameters
	if(featureName == null){
	    throw new IllegalArgumentException("SensoryItem name was null.");
	}
		
	List<SensoryItem> list = this.m_featuregroupTable.get(featureName);

	return list;
    }
	
    /** Returns a list of feature names that are in the AgentInputs
     * 
     * @return A list of feature names that are in the AgentInputs
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public List<String> getSensoryItemNames(){
	//get the list of feature types and make them into a list
	Set<String> names = this.m_featuregroupTable.keySet();
	ArrayList<String> nameList = new ArrayList<String>(names);
		
	return nameList;
    }
	
    /** Overrides the equals method
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    @Override
	public boolean equals(Object o){
	//test if same object
	if(this == o){
	    return true;
	}
	//test null or different class
	if(o == null || !(o instanceof AgentInputs)){
	    return false;
	}
		
	//typecase as this type of object
	AgentInputs v = (AgentInputs)o;
		
	//make sure they have same number of items
	if(this.getNumberSensoryItems() != v.getNumberSensoryItems()){
	    return false;
	}
		
	//check the types of features
	List<String> l1 = this.getSensoryItemNames();
	List<String> l2 = v.getSensoryItemNames();
	//make sure they contain same number of features
	if(l1.size() != l2.size()){
	    return false;
	}
		
	//make sure all the same type of features exist
	for(String featureName:l1){
	    if(!l2.contains(featureName)){
		return false;
	    }
	    //make sure they have the same number of this feature type
	    if(this.getNumberSensoryItems(featureName) != v.getNumberSensoryItems(featureName)){
		return false;
	    }
	    //make sure each feature has a match in the other list
	    List<SensoryItem> fl1 = this.getSensoryItems(featureName);
	    List<SensoryItem> fl2 = v.getSensoryItems(featureName);
	    for(SensoryItem f:fl1){
		if(!fl2.contains(f)){
		    return false;
		}
	    }
	}
		
	return true;
    }


    /** Prints out the difference between this instance and another one.
     * 
     * @param o The other instance of AgentInputs
     *
     * @author Edgar Acosta
     * @since 0.5
     */
    public void Diff(AgentInputs o){
	String indent="      ";
	if(this.getNumberSensoryItems() != o.getNumberSensoryItems()){
	    System.out.println(indent+"Difference in the number of Sensory Items");
	}
	List<String> li1=this.getSensoryItemNames();
	List<String> li2=o.getSensoryItemNames();
	String FirstIndent=indent+"  >";
	String SecondIndent=indent+"  <";
	for(String featureName:li1){
	    System.out.println(indent+featureName);
	    List<SensoryItem> SI1=this.getSensoryItems(featureName);
	    int oc=0;
	    for(SensoryItem si:SI1){
		oc++;
		String c=Integer.toString(oc);
		System.out.println(FirstIndent+" "+c+":"+si.getFeatureName());
	    }
	    oc=0;
	    if(li2.contains(featureName)){
		List<SensoryItem> SI2=o.getSensoryItems(featureName);
		for(SensoryItem si:SI2){
		    oc++;
		    String c=Integer.toString(oc);
		    System.out.println(SecondIndent+" "+c+":"+si.getFeatureName());
		}
		li2.remove(featureName);
	    }
	}
	for(String featureName:li2){
	    System.out.println(indent+featureName);
	    List<SensoryItem> SI2=o.getSensoryItems(featureName);
	    int oc=0;
	    for(SensoryItem si:SI2){
		oc++;
		String c=Integer.toString(oc);
		System.out.println(SecondIndent+" "+c+":"+si.getFeatureName());
	    }
	}
    }
}
