/** Case.java in the package org.JIFSA of the JIFSA project.
    Originally created 17-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    * 
    */

package org.JIFSA;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.JIFSA.reasoning.distance.GlobalDistanceMeasure;

/** A Case represents the current inputs to the agent (through vision or
 * other sensors) and the associated action(s) the agent takes.
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class Case implements Serializable {

    private static final long serialVersionUID = -7056509620159922081L;

    //the algorithm used to calculate distance between Cases
    private static GlobalDistanceMeasure m_globaldistancemeasure = null;
	
    /** What the agent observed */
    private AgentInputs m_inputs;
	
    /** The action(s) the agent performed*/
    private List<AgentAction> m_actions;
	
	
    /** The default constructor for Case objects.
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public Case() {
	//create an empty vision and empty actions
	this.m_inputs = new AgentInputs();
	this.m_actions = new ArrayList<AgentAction>();
    }
	
    /** Creates a Case based on the supplied AgentInputs and AgentActions
     * 
     * @param inputs The AgentInputs for the Case
     * @param action The AgentActions for the Case
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public Case(AgentInputs inputs, List<AgentAction> actions) {
	//check the parameters
	if(inputs == null || actions == null){
	    throw new IllegalArgumentException("Null AgentInputs or AgentActions list given when creating a Case");
	}
		
	this.m_inputs = inputs;
	this.m_actions = actions;
    }
	
	
    /** Sets the AgentActions associated with this Case
     * 
     * @return The associated action in this Case
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void setActions(List<AgentAction> actions){
	//TODO making a copy may be a performance issue
	this.m_actions= new ArrayList<AgentAction>(actions);
    }
	
    /** Returns the AgentActions associated with this Case
     * 
     * @return The associated action in this Case
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public List<AgentAction> getActions(){
	//TODO making a copy may be a performance issue
	return new ArrayList<AgentAction>(this.m_actions);
    }
	
    /** Returns the AgentInputs associated with this Case
     * 
     * @return The associated inputs in this Case
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public AgentInputs getInputs(){
	return this.m_inputs;
    }
	
    /** Sets the AgentInputs associated with this Case
     * 
     * @return The associated inputs in this Case
     *
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void setInputs(AgentInputs inputs){
	this.m_inputs=inputs;
    }
	
    /** Override of the equals method.
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    @Override
	public boolean equals(Object o){
	//see if its the same item
	if(this == o){
	    return true;
	}
		
	//check for null and other classes
	if(o == null){
	    return false;
	}
	if( !(o instanceof Case)){
	    return false;
	}
		
	//typecast the object as a case and get its components
	Case c = (Case)o;
	AgentInputs v = c.getInputs();
	List<AgentAction> l = c.getActions();
		
	//check to make sure the lists are the same
	if(l.size() != this.m_actions.size()){
	    return false;
	}
	for(int ii=0;ii<l.size();ii++){
	    //make sure each action is equal
	    if(!l.get(ii).equals(this.m_actions.get(ii)) ){
		return false;
	    }
	}
		
	//make sure the AgentVisions are equal
	if(!v.equals(this.m_inputs)){
	    return false;
	}
		
	return true;
    }
	
    /** Sets the algorithm that will be used to calculate the distance 
     * between Cases.
     * 
     * @param gdm The algorithm to use
     * 
     * @author Michael W. Floyd
     * @since 0.3
     */
    public static void setGlobalDistanceCalculation(GlobalDistanceMeasure gdm){
	//check parameter
	if(gdm == null){
	    throw new IllegalArgumentException("Null GlobalDistanceMeasure given to Case");
	}
		
	Case.m_globaldistancemeasure = gdm;
    }
	
    /** Returns the GlobalDistanceMeasure used by Cases
     * 
     * @return the currently set GlobalDistanceMeasure
     * 
     * @author Michael W. Floyd
     * @since 0.5
     */
    public static GlobalDistanceMeasure getGloablDistanceCalculation(){
	return Case.m_globaldistancemeasure;
    }
	

	
    /** Used to calculate the distance between two Cases. The algorithm
     * used to calculate pairwise distances between Cases must have
     * been previously set using the setGlobalDistanceCalculation 
     * method.
     * 
     * @param c1 The first Case
     * @param c2 The second Case
     * @return The distance between the Cases
     * 
     * @author Michael W. Floyd
     * @since 0.3
     */
    public static float pairwiseDistance(Case c1, Case c2){
	//check the parameters
	if(c1 == null || c2 == null){
	    throw new IllegalArgumentException("Null Case given to pairwiseDistance method.");
	}
		
	//make sure the algorithm has been set
	if(Case.m_globaldistancemeasure == null){
	    throw new IllegalStateException("The Case global distance algorithm has not been set.");
	}
		
	return Case.m_globaldistancemeasure.pairwiseDistance(c1, c2);
    }


    /** Prints out the difference between this instance and another one.
     * 
     * @param o The other instance of Case
     *
     * @author Edgar Acosta
     * @since 0.5
     */
    public void Diff(Case o){
	String indent="    ";
	AgentInputs Is1=this.getInputs();
	AgentInputs Is2=o.getInputs();
	if(! Is1.equals(Is2)){
	    System.out.println(indent+"Inputs difference");
	    Is1.Diff(Is2);
	}
	List<AgentAction> As1=this.getActions(); 
	List<AgentAction> As2=o.getActions();
	
	if(As1.size() != As2.size()){
	    System.out.println(indent+"Difference: in the number of actions"+
			       Integer.toString(As1.size())+" "+
			       Integer.toString(As2.size()));
	} else{
	    for(int ii=0;ii<As1.size();ii++){
		//make sure each action is equal
		if(!As1.get(ii).equals(As2.get(ii)) ){
		    System.out.println(indent+"Action difference");
		}
	    }
	}
    }
}
