/** CaseBase.java in the package org.JIFSA of the JIFSA project.
    Originally created 18-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    * 
    */

package org.JIFSA;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

/** The case base that is used by the agent for case-based reasoning.
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class CaseBase implements Serializable {
	
    private static final long serialVersionUID = -5826431951330607037L;
	
    /** data collection to store cases */
    private List<Case> m_casebase;
	
    /** The default constructor for the CaseBase object.
     * 
     * @author Michael W. Floyd
     * @since 1.0
     */
    public CaseBase(){
	this.m_casebase = new ArrayList<Case>();
    }
	
    /** Copy Constructor
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public CaseBase(CaseBase cb){
	//check params
	if(cb == null){
	    throw new IllegalArgumentException("Null paramater given to CaseBase constructor");
	}
		
	List<Case> caseList = cb.getCaseList();
	this.m_casebase = new ArrayList<Case>(caseList);
    }
	
    /** Returns the current size of the casebase.
     * 
     * @return The current size of the case base
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public int getCasebaseSize(){
	return this.m_casebase.size();
    }
	
    /** Adds a Case to the CaseBase
     * 
     * @param ca The Case to add to the CaseBase
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void addCase(Case ca){
	if(ca == null){
	    throw new IllegalArgumentException("Null Case added to CaseBase.");
	}
		
	this.m_casebase.add(ca);
    }
	
    /** Returns a list of Cases in the CaseBase
     * 
     * @return The list of Cases in the CaseBase
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public List<Case> getCaseList(){
	return new ArrayList<Case>(this.m_casebase);
    }
	
    /** Sets the list of Cases in the CaseBase
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    public void setCaseList(List<Case> cbList){
	//check params
	if(cbList == null){
	    throw new IllegalArgumentException("Null paramater given to CaseBase constructor");
	}
		
	this.m_casebase = new ArrayList<Case>(cbList);
    }
	
    /** Overrides the equals method.
     * 
     * @author Michael W. Floyd
     * @since 0.1
     */
    @Override
	public boolean equals(Object o){
	//see if it is the same object
	if(this == o){
	    return true;
	}
		
	//check to see if null or not of right type
	if(o == null){
	    return false;
	}
	if( !(o instanceof CaseBase)){
	    return false;
	}
		
	//typecast the object
	CaseBase cb = (CaseBase)o;
		
	//check the size
	if(this.getCasebaseSize() != cb.getCasebaseSize()){
	    return false;
	}
		
	//make sure the same cases are in each casebase
	for(Case c: cb.getCaseList()){
	    if(!this.m_casebase.contains(c)){
		return false;
	    }
	}
		
	//otherwise return true
	return true;
    }
	

    /** Prints out the difference between this instance and another one.
     * 
     * @param o The other instance of CaseBase
     *
     * @author Edgar Acosta
     * @since 0.5
     */
    public void Diff(CaseBase o){
	List fCBl=this.getCaseList();
	List sCBl=o.getCaseList();
	ListIterator fCBi = fCBl.listIterator();
	ListIterator sCBi = sCBl.listIterator();
	int CaseCounter=0;
	while(fCBi.hasNext()){
	    CaseCounter++;
	    String cc=Integer.toString(CaseCounter);
	    if(sCBi.hasNext()){
		Case fC=(Case)fCBi.next();
		Case sC=(Case)sCBi.next();
		if(fC.equals(sC)){
		    System.out.println("  Case "+cc+" is ok.");
		} else {
		    System.out.println("  Case "+cc);
		    fC.Diff(sC);
		}
	    }
	    else{
		System.out.println("  The first Case Base has more cases than the second one.");
	    }
	}
	if(sCBi.hasNext()){
	    System.out.println("  The second Case Base has more cases than the first one.");
	}
    }
}
