/** SensoryItem.java in the package org.JIFSA of the JIFSA project.
    Originally created 29-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.JIFSA;

import java.io.Serializable;

import org.JIFSA.reasoning.distance.DistanceCalculation;
import org.JIFSA.reasoning.distance.penalty.PenaltyDistance;

/** This class is a general representation of a feature in a Case.
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class SensoryItem implements Serializable {

	private static final long serialVersionUID = -6837660342788953122L;
	
	//plug and play algorithms used for calculations for the entire class
	private static DistanceCalculation m_distanceCalculation = null;
	private static PenaltyDistance m_penaltyCalculation = null;
	
	/** The name of the feature */
	private String m_featureName;
	
	/** The constructor given the name of the feature
	 * 
	 * @param featureName The name of the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	public SensoryItem(String featureName){
		//check parameters
		if(featureName == null){
			throw new IllegalArgumentException("Null name when creating a SensoryItem.");
		}
		this.m_featureName = featureName;
	}
	
	/** Returns the name of the feature
	 * 
	 * @return The name of the feature
	 *
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	public String getFeatureName(){
		return this.m_featureName;
	}
	
	/** Overrides the equals method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Override
	public boolean equals(Object o){
		//see if this is the object
		if(this == o){
			return true;
		}
		
		//test for null or other class type
		if(o == null || !(o instanceof SensoryItem)){
			return false;
		}
		
		//typecast the object
		SensoryItem f = (SensoryItem)o;
		
		//we compare action names
		return this.m_featureName.equals(f.getFeatureName());	
	}
	
	/** Used to set the algorithm that will be used to calculate
	 * the distance between features of this type (or possible features
	 * of a subclass)
	 * 
	 * @param dc The distance calculation to use
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	public static void setDistanceCalculation(DistanceCalculation dc){
		//check the validity of parameters
		if(dc == null){
			throw new IllegalArgumentException("Distance calculation algorithm cannot be null.");
		}
		SensoryItem.m_distanceCalculation = dc;
	}
	
	/** Removes the distance calculation algorithm.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public static void removeDistanceCalculation(){
		SensoryItem.m_distanceCalculation = null;
	}
	
	/** Calculates the distance between two features. The distance calculation algorithm
	 * used must be set using the SensoryItem.setDistanceCalculation(DistanceCalculation) method
	 * or else an exception will be thrown.
	 * 
	 * @param f The other feature
	 * @return the distance betweent the two features
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	public float pairwiseDistance(SensoryItem f){
		//check the validity of parameters
		if(f == null){
			throw new IllegalArgumentException("A null feature was given.");
		}
		
		//make sure we have a distance calculation algorithm set
		if(SensoryItem.m_distanceCalculation == null){
			throw new IllegalStateException("The distance calculation algorithm has not yet been set.");
		}
		
		//if everything is fine, return the pairwise distance
		return SensoryItem.m_distanceCalculation.pairwiseDistance(this, f);
	}
	
	/** Sets the algorithm that will be used to calculate penalties.
	 * 
	 * @param pd The algorithm to use
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 * 
	 */
	public static void setPenaltyDistanceCalculation(PenaltyDistance pd){
		//check the validity of parameters
		if(pd == null){
			throw new IllegalArgumentException("Penalty calculation algorithm cannot be null.");
		}
		SensoryItem.m_penaltyCalculation = pd;
		
	}
	
	/** Removes the penalty distance calculation algorithm.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public static void removePenaltyDistanceCalculation(){
		SensoryItem.m_penaltyCalculation = null;
	}
	
	/** Calculates the distance penalty for the feature. This is
	 * usually used when a SensoryItem has no matching feature in another
	 * case (unequal number of features).
	 * 
	 * @return The penalty
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public float penaltyDistance(){
	
		//make sure we have a penalty calculation algorithm set
		if(SensoryItem.m_penaltyCalculation == null){
			throw new IllegalStateException("The penalty calculation algorithm has not yet been set.");
		}
		
		//if everything is fine, return the pairwise distance
		return SensoryItem.m_penaltyCalculation.calculatePenalty(this);
	}
	
	
	/** Returns an XML representation of the SensoryItem
	 * 
	 * @return The SensoryItem in XML format
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public String toXML(){
		String xml = "";
		
		xml += "<SensoryItem>";
		xml += "<FeatureName>";
		xml += this.m_featureName;
		xml += "</FeatureName>";
		xml += "</SensoryItem>";
		
		return xml;
	}
	
}
