/** StandardEvaluation.java in the package org.JIFSA.preprocessing.featureselection of the JIFSA project.
    Originally created 16-Feb-08

    Copyright (C) 2008  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.JIFSA.preprocessing.featureselection;


import java.util.List;

import org.JIFSA.Agent;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.performance.ClassificationStatisticsWrapper;
import org.JIFSA.performance.StatisticsBundle;
import org.JIFSA.performance.StatisticsWrapper;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.actionselection.actionestimation.ActionEstimation;
import org.JIFSA.tools.RandomCaseBase;

/** This class performs a standard evaluation of the feature weights using a fixed
 * sized training case base.
 * 
 * @author Michael W. Floyd
 * @since 0.5
 */
public class StandardEvaluation implements FeatureWeightEvaluation {

	/** used to evaluate the weights */
	private Agent m_evaluationAgent;
	private CaseBase m_training;
	private CaseBase m_testing;
	private ActionEstimation m_ae;
	
	/** Creates the evaluation function that is used to evaluate a set of weights.
	 * 
	 * @param a The Agent that is using the weights
	 * @param ae The method of action estimation that is used (to determine if matches are correct)
	 * @param allCases A case base containing the entire set of cases
	 * @param numTraining The number of cases from allCases that should be used for the training set
	 * @param numTesting The number of cases from allCases that should be used for the testing set
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public StandardEvaluation(Agent a, ActionEstimation ae, CaseBase allCases, int numTraining, int numTesting){
		if(a == null){
			throw new IllegalArgumentException("Null agent.");
		}
		if(ae == null){
			throw new IllegalArgumentException("Null ActionEstimation.");
		}
		if(allCases == null){
			throw new IllegalArgumentException("Null CaseBase.");
		}
		if(numTraining <= 0 || numTraining > allCases.getCasebaseSize()){
			throw new IllegalArgumentException("Invalid number of training Cases specified.");
		}
		if(numTesting <= 0 || numTesting > allCases.getCasebaseSize()){
			throw new IllegalArgumentException("Invalid number of training Cases specified.");
		}
		
		m_evaluationAgent = a;
		m_ae = ae;
		m_training = RandomCaseBase.random(allCases,numTraining);
		m_evaluationAgent.setCaseBase(m_training);
		m_testing = RandomCaseBase.random(allCases,numTesting);
	}
	
	/** Evaluates the supplied weights and returns the statistics generated
	 * when using those weights.
	 * 
	 * @return the statistics when using the supplied weights
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public StatisticsBundle evaluate(Weights ow) {
		if(ow == null){
			throw new IllegalArgumentException("Null object weights.");
		}
		
		//set the new weights to use
		Case.getGloablDistanceCalculation().setWeights(ow);
		
		
		List<Case> testingList = m_testing.getCaseList();
		
		StatisticsWrapper evaluationFunction = new ClassificationStatisticsWrapper(m_evaluationAgent, m_ae);
		
		//give the system each of the evaluation cases
		for(Case c : testingList){
			evaluationFunction.senseEnvironment(c);
		}
		
		return evaluationFunction.getStatisticsBundle();
	}

}
