/** SequentialBackwardGeneration.java in the package org.JIFSA.preprocessing.featureselection.algorithms of the JIFSA project.
    Originally created 16-Feb-08

    Copyright (C) 2008  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.JIFSA.preprocessing.featureselection.algorithms;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.preprocessing.featureselection.FeatureWeightEvaluation;

import org.JIFSA.reasoning.Weights;
import org.JIFSA.performance.StatisticsBundle;

/** The Sequential Backward Generation wrapper algorithm described in
 * 
 * R. Kohavi, Wrapper for Performance Enhancement and Oblivious Decision
 * Graphs, PhD thesis, Stanford University, 1995.
 * 
 * The algorithm initially starts with with a complete feature set and then
 * "opens" that set by examining all possible feature sets that exist by removing
 * a single feature. The feature set with the highest evaluation is likewise
 * "opened". This process continues until there is not a significant increase 
 * in the evaluation of feature sets for k "openings".
 * 
 * @author Michael W. Floyd
 * @since 0.5
 */
public class SequentialBackwardGeneration implements WrapperFSA {

	/** the feature set with highest evaluation */
	private Weights m_best;
	/** the evaluation of the best */
	private float m_evalBest;
	/** the stats of the best */
	private StatisticsBundle m_statsBest;
	/** the feature sets that have already been considered */
	private List<Weights> m_closed;
	/** the feature sets that have been opened but not considered */
	private List<Weights> m_open;
	/** the evaluations of the open feature sets */
	private List<Float> m_evalOpen;
	/** the full stats for the open feature set */
	private List<StatisticsBundle> m_statsOpen;
	/** the maximum number of non-improving openings */
	private int m_k;
	/** the amount by which a node must improve over the best */
	private double m_multiplier;
	/** the evaluation algorithm */
	private FeatureWeightEvaluation m_fwe;

	
	
	/** Constructs the algorithm. 
	 * 
	 * @param k The maximum number of nodes to open without an improvement
	 * @param epsilon The percentage by which the current node must improve on the best node (a value of 1.0 means a 1% increase).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public SequentialBackwardGeneration(int k, double epsilon){
		//check parameters
		if(k < 1){
			throw new IllegalArgumentException("k-value must be greater than zero.");
		}
		if(epsilon < 0){
			throw new IllegalArgumentException("epsilon must be positive.");
		}
		
		m_k = k;
		m_multiplier = 1 + (epsilon/100);
		
		m_best = null;
		m_evalBest = -1;
		m_statsBest = null;
		m_closed = new ArrayList<Weights>();
		m_open = new ArrayList<Weights>();
		m_evalOpen = new ArrayList<Float>();
		m_statsOpen = new ArrayList<StatisticsBundle>();
		
		m_fwe = null;
	}
	
	/** Performs Sequential Backward Feature Selection to find the set of weights that
	 * maximize an evaluation function.
	 * 
	 * @param fwe The evaluation function
	 * @param allIncluded A set of weights that includes 1.0 weightings for all features
	 * @return The set of weights that maximize the evaluation function
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public Weights selectFeatures(FeatureWeightEvaluation fwe, Weights allIncluded) {
		//check params
		if(fwe == null){
			throw new IllegalArgumentException("Null FeatureWeightEvaluation given.");
		}
		
		m_fwe = fwe;
		
		
		//this is the only one, so it must be the best
		m_statsBest = m_fwe.evaluate(allIncluded);
		m_evalBest = m_statsBest.getPrimaryStatistic();
		m_best = allIncluded;
		

		//the weights we are currently dealing with
		Weights currentWeight = allIncluded;
		
		//we use this to store the number of weight we have "opened" since
		//the best weight set
		int numSinceBest = 0;
		//now we begin the best-first search
		while(numSinceBest < m_k){
			
			//we have already examined this set, so we will add it to closed
			m_closed.add(currentWeight);
			
			//find the children of the current feature set and add any of the children
			//that have not been seen yet to the open list
			determineChildren(currentWeight);
			
			//make sure more values exist, otherwise we are done
			if(m_open.size() == 0){
				break;
			}
			
			//get the highest in the list
			currentWeight = m_open.remove(0);
			float currentEval = m_evalOpen.remove(0);
			StatisticsBundle currentStats = m_statsOpen.remove(0);
			
			//see if it is better than a certain percentage
			if(currentEval > (m_evalBest*m_multiplier) ){
				//we have a new best
				m_best = currentWeight;
				m_statsBest = currentStats;
				m_evalBest = currentEval;
				numSinceBest = 0;	
			}else{
				numSinceBest++;
			}
			
		}
		
	
		return m_best;
	}
	
	
	/** This method determines the children of the current 
	 * feature set by finding all children with one less
	 * feature
	 * 
	 * @param current The current feature set being examined
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	private void determineChildren(Weights current){
		//get the weights of the current
		List<String> weights = current.getWeightedItemNames();
		
		//go through each weight and remove if necessary
		for(String currentWeightName : weights){
			//ignore if the weight has already been removed
			if( current.getWeight(currentWeightName)> 0){
				//copy the weights, with any unspecified weight having a weight of zero
				Weights newChild = new Weights(0.0f);
				for(String nextWeight : weights){
					newChild.setWeight(nextWeight, current.getWeight(nextWeight));
				}
				
				//change the current weight to a zero
				newChild.setWeight(currentWeightName, 0.0f);
				
				//we only want to add it to a list if we haven't seen this weighting before
				if(!m_closed.contains(newChild) && !m_open.contains(newChild)){
					System.out.println("Evaluating new child: " + newChild);
					StatisticsBundle stats = m_fwe.evaluate(newChild);
					float evaluation = stats.getPrimaryStatistic();
					placeInList(newChild,evaluation,stats);
				}
			}
		}
	}
	
	/** Places the weight set, evaluation and other stats in their respective lists. They
	 * are placed in order so that the weight set with the highest evaluation is
	 * first in the list.
	 * 
	 * @param child The weight set
	 * @param eval The evaluation of that weight set
	 * @param stats The other stats of the weight set
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	private void placeInList(Weights child, float eval, StatisticsBundle stats){
		
		boolean placed = false;
		
		for(int ii=0;ii<m_evalOpen.size();ii++){
			//if it has a higher evaluation that the current item, add it here
			if(eval > m_evalOpen.get(ii)){
				m_evalOpen.add(ii, new Float(eval));
				m_statsOpen.add(ii,stats);
				m_open.add(ii, child);
				placed = true;
				break;
			}
		}
		
		//if it was not placed somewhere in the list, add to the end
		if(!placed){
			m_evalOpen.add(new Float(eval));
			m_statsOpen.add(stats);
			m_open.add(child);
		}
	}

	/** Returns the statistics for the current best weighting that has been found by the algorithm.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	public StatisticsBundle getStatisticsOfBestWeights() {
		return m_statsBest;
	}


}
