/** FeatureDistanceCasePresort.java in the package org.JIFSA.preprocessing.filters.casefilter of the JIFSA project.
    Originally created 25-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.JIFSA.preprocessing.filters.casefilter;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.JIFSA.preprocessing.filters.CaseFilter;
import org.JIFSA.sensoryItems.Spatial2DObject;

/** Sorts a list of Spatial2DObjects by their distance to the
 *  agent.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class FeatureDistanceCasePresort implements CaseFilter {

	/** Takes in an initial Case and sorts all Spatial2DFeature objects
	 * by their distance from the agent.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public Case filter(Case initialCase) {
		//check parameters
		if(initialCase == null){
			throw new IllegalArgumentException("Null case given to FeatureDistanceCasePresort.");
		}
		
		//get the data from the original Case
		AgentInputs initialVision = initialCase.getInputs();
		List<AgentAction> actions = initialCase.getActions();
		
		//create a new AgentInputs
		AgentInputs newVision = new AgentInputs();
		
		//go through each feature type and sort the SensoryItem list
		List<String> featureNames = initialVision.getSensoryItemNames();
		for(String currentFeature : featureNames){
			List<SensoryItem> currentList = initialVision.getSensoryItems(currentFeature);
			
			//if this type of features are Spatial2DObjects, sort them
			if(currentList != null){
				if(currentList.size() > 0){
					if(currentList.get(0) instanceof Spatial2DObject){
						sortByDistance(currentList);
					}
				}
				
				//now add the sorted list to the new vision
				for(SensoryItem currentFeat : currentList){
					newVision.addSensoryItem(currentFeat);
				}
			}
		}
		
		//now make it into the new sorted feature case
		Case newCase = new Case(newVision, actions);
		return newCase;
	}
	
	/** Sorts a list of Spatial2DObjects by their distance to the
	 * agent.
	 * 
	 * @param sensoryItemList A list of Spatial2DFeatures
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	private void sortByDistance(List<SensoryItem> sensoryItemList){
		
		//define a comparator for sorting a List of Features by distance
		Comparator<SensoryItem> comp = new Comparator<SensoryItem>(){
			 public int compare(SensoryItem f1, SensoryItem f2){
				 
				 Spatial2DObject sf1 = (Spatial2DObject)f1;
				 Spatial2DObject sf2 = (Spatial2DObject)f2;
				 
				 float dist1 = sf1.getDistance();
				 float dist2 = sf2.getDistance();
				 
				 if (dist1 == dist2){
					 return 0;
				 }else if(dist1 > dist2){
					 return 1;
				 }else{
					 return -1;
				 }
			 }
		};
		
		//sort the List based on our comparator we defined
		Collections.sort(sensoryItemList, comp);
	}

}
