/** Weights.java in the package org.JIFSA.reasoning of the JIFSA project.
    Originally created 21-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.JIFSA.reasoning;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;


/** This class represents weights that are are applied to items. The initial
 * design of this class is for SensoryItem and AgentAction weights, although
 * it can be applied to any situation where a weighting is needed.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class Weights {

	//store the default weight
	private float m_defaultWeight;
	//store the set weights
	private Map<String,Float> m_weights;
	
	/** Creates a new Weights object. The parameter
	 * represents the default weight given to an
	 * item that has not been set with the setWeight
	 * function. The default weight must be between 
	 * 0.0 and 1.0 (inclusive).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public Weights(float defaultWeight){
		//check the parameters
		if(defaultWeight < 0.0f || defaultWeight > 1.0f){
			throw new IllegalArgumentException("The weights must be between 0.0 and 1.0 (inclusive)");
		}
		
		this.m_defaultWeight = defaultWeight;
		this.m_weights = new HashMap<String,Float>();
	}
	
	/** Sets the weight for the given type of item
	 * to the designated value. The weight value must
	 * be between 0.0 and 1.0 (inclusive).
	 * 
	 * @param itemName The name of the item type
	 * @param weight The weight for that item type
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public void setWeight(String itemName, float weight){
		//check parameters
		if(itemName == null){
			throw new IllegalArgumentException("Null parameter given to Weights object.");
		}
		if(weight < 0.0f || weight > 1.0f){
			throw new IllegalArgumentException("The weights must be between 0.0 and 1.0 (inclusive)");
		}
		
		//set the weight value
		this.m_weights.put(itemName, new Float(weight));
	}
	
	/** Returns a list of all types of items that
	 * have had their weights explicitly set with
	 * the setWeight function.
	 * 
	 * @return The list of item types
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public List<String> getWeightedItemNames(){
		Set<String> col = this.m_weights.keySet();
		List<String> itemList = new ArrayList<String>(col);
		return itemList;
	}
	
	/** Returns the weight for the given type
	 * of item. If that weight has not been
	 * set using the setWeight function then
	 * the default weight value given by
	 * the constructor will be used.
	 * 
	 * @param itemName The type of item
	 * @return The weight of the item
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public float getWeight(String itemName){
		//check parameters
		if(itemName == null){
			throw new IllegalArgumentException("Null parameter given to Weights object.");
		}
		
		//get the associated weight
		Float weight = this.m_weights.get(itemName);
		
		//see if the weight has actually been set, otherwise use default
		if(weight == null){
			return this.m_defaultWeight;
		}
		return weight.floatValue();
	}
	
	/** Overrides the equals method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Override
	public boolean equals(Object o){
		//see if its the same item
		if(this == o){
			return true;
		}
		
		//check for null and other classes
		if(o == null){
			return false;
		}
		if( !(o instanceof Weights)){
			return false;
		}
		
		//typecast the object as a case and get its components
		Weights w = (Weights)o;
		
		if(this.m_defaultWeight != w.m_defaultWeight){
			return false;
		}
		if(!this.m_weights.equals(w.m_weights)){
			return false;
		}
		
		return true;
	}
	
	/** Overrides the toString method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Override
	public String toString(){
		String str = "{";
		
		List<String> wts = this.getWeightedItemNames();
		for(String nextwt : wts){
			if(wts.indexOf(nextwt) != 0){
				str += ",";
			}
			
			str += (nextwt + ":" + this.getWeight(nextwt));
		}
		
		str += "}";
		
		return str;
	}
}
