/** NearestNeighbourSearch.java in the package org.JIFSA.reasoning.casebasesearch of the JIFSA project.
    Originally created Nov 23, 2007

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.JIFSA.reasoning.casebasesearch;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;

/** Used to perform a k-Nearest Neighbour search to find the
 * nearest neighbours to a specified Case.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class NearestNeighbourSearch implements CaseBaseSearch {

	//the number of neighbours
	private int m_numNeighbours;
	
	/** Creates a NearestNeighbourSearch that uses the specified
	 * distance measure to determine how near a neighbour is and locates
	 * the specified number of nearest neighbours.
	 * 
	 * @param distance The distance calculation algorithm to use
	 * @param k The number of nearest neighbours to return
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public NearestNeighbourSearch(int k){
		//check parameters
		if(k <= 0){
			throw new IllegalArgumentException("Invalid number of neighbours given to NearestNeighbourSearch");
		}
		
		this.m_numNeighbours = k;
	}
	
	/** Finds a specific number of nearest neighbours (as specified in
	 * constructor) and returns them. If multiple Cases are equally close
	 * but there is not room to return all of them, the Case that is found
	 * first will be returned.
	 * 
	 */
	public List<Case> findClosest(Case currentCase, CaseBase possibleCases) {
		//check parameters
		if(currentCase == null){
			throw new IllegalArgumentException("Null Case given to NearestNeighbourSearch");
		}
		if(possibleCases == null){
			throw new IllegalArgumentException("Null CaseBase given to NearestNeighbourSearch");
		}
			
		//set to defaults the nearest neighbours and their distances
		List<Case> nearest = new ArrayList<Case>(this.m_numNeighbours);
		List<Float> nearDistances = new ArrayList<Float>(this.m_numNeighbours);
		for(int ii=0; ii < this.m_numNeighbours; ii++){
			nearest.add(null);
			//the null values are as far as possible away
			nearDistances.add(new Float(Float.MAX_VALUE));
		}
		
		//retrieve the Cases in CaseBase and go through them
		List<Case> caseBase = possibleCases.getCaseList();
		for(Case nextCase : caseBase){
			float nextDistance = Case.pairwiseDistance(currentCase, nextCase);
			for(int ii=0; ii < this.m_numNeighbours; ii++){
				if(nextDistance < nearDistances.get(ii).floatValue()){
					//since it is closer, put the Case at this position
					nearest.add(ii,nextCase);
					nearDistances.add(ii, new Float(nextDistance));
					
					//remove the highest value since we added a lower value
					nearest.remove(this.m_numNeighbours);
					nearDistances.remove(this.m_numNeighbours);
					//we added it, so we dont need to search the rest
					break;
				}
			}
			
		}
		
		//the null values were used as place holders, we can remove any of them
		while(nearest.remove(null)){
			//Do nothing here, we just want to remove the nulls
		}
		
		return nearest;
	}

}
