/** GlobalDistanceMeasure.java in the package org.JIFSA.reasoning.distance of the JIFSA project.
    Originally created Nov 23, 2007

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.JIFSA.reasoning.distance;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.JIFSA.Case;
import org.JIFSA.reasoning.Weights;

/** Given two Cases, will calculate the distance between them.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public abstract class GlobalDistanceMeasure {
	
	protected Weights m_featureWeights;

	/** Creates a global distance measure that uses the provided
	 * feature weights.
	 * 
	 * @param weights The feature weights
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public GlobalDistanceMeasure(Weights featureWeights){
		//check params
		if(featureWeights == null){
			throw new IllegalArgumentException("Null Weights given to GlobalDistanceMeasure");
		}
		
		this.m_featureWeights = featureWeights;
	}
	
	
	/** Computes the distance between two Cases.
	 * 
	 * @param c1 The first case
	 * @param c2 The second case
	 * @return The distance between the cases
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public abstract float pairwiseDistance(Case c1, Case c2);
	

	/** Sets the feature weights to the given weights.
	 * 
	 * @param newWeights The weights to use
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public void setWeights(Weights newWeights){
		if(newWeights == null){
			throw new IllegalArgumentException("Null weights given to GlobalDistanceMeasure");
		}
		this.m_featureWeights = newWeights;
	}
	
	/** Gets the feature weights currently used by the global distance
	 * measure
	 * 
	 * @return The feature weights
	 * @since 0.3
	 */
	public Weights getWeights(){
		return this.m_featureWeights;
	}
	
	/** Based on the weights of each type of feature, a complete distance
	 * will be produced.
	 * 
	 * @param distances The distance of each type of feature
	 * @return The weighted distance
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	protected float weightedDistance(Map<String, Float> distances) {
		float totalDistance = 0.0f;
		
		//get a list of the type of features we have and go through each type
		List<String> featureTypes = new ArrayList<String>(distances.keySet());
		for(String currentFeature : featureTypes){
			float featureDistance = distances.get(currentFeature);
				
			//apply the weight
			float currentWeight = this.m_featureWeights.getWeight(currentFeature);
			totalDistance += featureDistance*currentWeight;
		}
		
		return totalDistance;
	}
}
