/** OrderIndexMatchingAlgorithm.java in the package org.JIFSA.reasoning.distance.globaldistance of the JIFSA project.
    Originally created 16-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.JIFSA.reasoning.distance.globaldistance;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.distance.GlobalDistanceMeasure;



/** Assumes features are sorted in a meaningful way. Features are paired with the
 * SensoryItem of the same type, with the same index, in the other Case. For example,
 * if each Case had two Features (all the same type of features) then then SensoryItem
 * closest to the agent in the first case would be paired with the SensoryItem closest 
 * to the agent in the second case (assuming they have been presorted by distance). 
 * As well, the two other Features would be matched together. 
 * 
 * If there are an uneven number of Features of a given type, then remaining
 * Features will be matched with a null value.
 * 
 * In order for this algorithm to produce meaningful results, the CaseBase MUST have
 * the Features in the AgentInputs of each Case sorted in some meaningful way. This
 * will require applying a presort filter to the CaseBase and a presort Case filter
 * to each new AgentInputs that is sensed (so both the CaseBase and new inputs are sorted
 * the same way).
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class OrderIndexMatchingAlgorithm extends GlobalDistanceMeasure {

	
	/** Creates an OrderIndexMatchingAlgorithm that uses the supplied
	 * feature weights. 
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public OrderIndexMatchingAlgorithm(Weights w){
		super(w);
	}
	
	/** 
	 * @see org.JIFSA.reasoning.distance.GlobalDistanceMeasure#pairwiseDistance(org.JIFSA.Case, org.JIFSA.Case)
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Override
	public float pairwiseDistance(Case c1, Case c2) {
		//check parameters
		if(c1 == null || c2 == null){
			throw new IllegalArgumentException("Null cases given to feature matching algorithm.");
		}
		
		AgentInputs case1Vision = c1.getInputs();
		AgentInputs case2Vision = c2.getInputs();
		
		//make a list of all feature types that exist in the cases
		List<String> featureTypes = case1Vision.getSensoryItemNames();
		List<String> secondFeatureTypes = case2Vision.getSensoryItemNames();
		
		//combine the lists together
		for(String feature: secondFeatureTypes){
			if(!featureTypes.contains(feature)){
				featureTypes.add(feature);
			}
		}
		
		//used to store the distances of each feature type
		Map<String,Float> distances = new HashMap<String,Float>();
		
		//now we go through each type of feature
		for(String feature: featureTypes){
			//get the features from each case
			List<SensoryItem> case1SensoryItems = case1Vision.getSensoryItems(feature);
			if(case1SensoryItems == null){
				case1SensoryItems = new ArrayList<SensoryItem>();
			}
			List<SensoryItem> case2SensoryItems = case2Vision.getSensoryItems(feature);
			if(case2SensoryItems == null){
				case2SensoryItems = new ArrayList<SensoryItem>();
			}
									
			//store the total distance and number of pairs for this feature type
			float totalDistance = 0;
			int numPairs = 0;
			
			//determine which list has more features in it
			int c1Size = case1SensoryItems.size();
			int c2Size = case2SensoryItems.size();
			boolean case1MoreFeatures = (c1Size > c2Size ? true : false );

			//now pair them
			if(case1MoreFeatures){
				//make pairs
				for(int ii=0; ii<c2Size; ii++){
					SensoryItem case1SensoryItem = case1SensoryItems.get(ii);
					totalDistance += case1SensoryItem.pairwiseDistance(case2SensoryItems.get(ii));
					numPairs++;
				}
				
				//now make pairs with the leftovers
				for(int jj = c2Size; jj<c1Size; jj++){
					SensoryItem case1SensoryItem = case1SensoryItems.get(jj);
					totalDistance += case1SensoryItem.penaltyDistance();
					numPairs++;
				}
			}else{
				//make pairs
				for(int ii=0; ii<c1Size; ii++){
					SensoryItem case2SensoryItem = case2SensoryItems.get(ii);
					totalDistance += case2SensoryItem.pairwiseDistance(case1SensoryItems.get(ii));
					numPairs++;
				}
				
				//now make pairs with the leftovers
				for(int jj = c1Size; jj<c2Size; jj++){
					SensoryItem case2SensoryItem = case2SensoryItems.get(jj);
					totalDistance += case2SensoryItem.penaltyDistance();
					numPairs++;
				}
			}
			
			float normalizedDistance = totalDistance/numPairs;
			
			//now we store the normalizedDistance;
			distances.put(feature, normalizedDistance);
		}
		
		return weightedDistance(distances);
	}
	
}
