/** PolarDistanceAlgorithm.java in the package org.JIFSA.reasoning.distance.spatial2D of the JIFSA project.
    Originally created 16-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.JIFSA.reasoning.distance.spatial2D;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.DistanceCalculation;
import org.JIFSA.sensoryItems.Spatial2DObject;

/** Calculates the polar distance between two features. This features must
 * be Spatial2DObjects that have a distance and direction from the agent.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class PolarDistanceAlgorithm implements DistanceCalculation {

	
	/** Calculates the pairwise distance between two features.
	 * 
	 */
	public float pairwiseDistance(SensoryItem f1, SensoryItem f2) {
		//check for null parameters
		if(f1 == null || f2 == null){
			throw new IllegalArgumentException("Null first feature given to polar distance algorithm.");
		}
		
		//make sure they are Spatial features (or the second is null)
		if( !(f1 instanceof Spatial2DObject) || !(f2 instanceof Spatial2DObject)){
			throw new IllegalArgumentException("Non-spatial features given to polar distance algorithm.");
		}
		Spatial2DObject sf1 = (Spatial2DObject)f1;
		Spatial2DObject sf2 = (Spatial2DObject)f2;
		
		//get the values from the features
		float r1 = sf1.getDistance();
		float th1 = sf1.getDirection();
		float r2 = sf2.getDistance();
		float th2 = sf2.getDirection();
		
		//dist^2 = r1^2 + r2^2 -2r1r2 cos (|th1-th2|)
		double theta = Math.abs(th2 - th1);
		double thetaRad = Math.toRadians(theta);
		double compoundPart = 2*r1*r2*Math.cos(thetaRad);
		double distSquared = r1*r1 + r2*r2 - compoundPart;
		
		return (float)Math.sqrt(distSquared);
	}

}
