/** Spatial2DObject.java in the package org.JIFSA.sensoryItems of the JIFSA project.
    Originally created 29-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.JIFSA.sensoryItems;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.DistanceCalculation;

/** This class represents a feature that exists at a 
 * 2-dimensional locations. The location is stored
 * as the distance and direction of the object relative
 * to the agent.
 * 
 * @author Michael W. Floyd
 * @since 0.2
 *
 */
public class Spatial2DObject extends SensoryItem {
	
	private static final long serialVersionUID = 2126001314974619011L;

	//the algorithm to use for distance calculation
	private static DistanceCalculation m_distanceCalculation = null;
	
	//the distance to the object
	private float m_distance;
	//the direction to the object
	private float m_direction;
	
	/** Creates a Spatial2DObject with the given feature name.
	 * The distance and direction are set to 0.
	 * 
	 * @param featureName The name of the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public Spatial2DObject(String featureName){
		super(featureName);
		//by default we say the objects are at our current location
		this.m_distance = 0;
		this.m_direction = 0;
	}
	
	/** Set the distance to the feature.
	 * 
	 * @param dist Distance to the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setDistance(float dist){
		this.m_distance = dist;
	}
	
	/** Set the direction to the feature.
	 * 
	 * @param dir Direction to the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setDirection(float dir){
		this.m_direction = dir;
	}
	
	/** The distance to the feature.
	 * 
	 * @return The distance to the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public float getDistance(){
		return this.m_distance;
	}
	
	/** The direction to the feature.
	 * 
	 * @return The direction to the feature
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public float getDirection(){
		return this.m_direction;
	}
	
	/** Sets the distance calculation algorithm that will be used on
	 * Features of this type.
	 * 
	 * @param dc The algorithm to use
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	public static void setDistanceCalculation(DistanceCalculation dc){
		//check the parameters
		if(dc == null){
			throw new IllegalArgumentException("Null DistanceCalculation given to Spatial2DObject");
		}
		Spatial2DObject.m_distanceCalculation = dc;
	}
	
	/** Removes the distance calculation.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public static void removeDistanceCalculation(){
		Spatial2DObject.m_distanceCalculation = null;
	}
	
	/** Calculates the pairwise distance between two features using the
	 * algorithm set using setDistanceCalculation. If that method has not
	 * been set, the distance will be calculated using the pairwise distance
	 * method from SensoryItem. Both Features must be Spatial2DObjects.
	 * 
	 * 
	 * @param f1 The other feature
	 * @return The distance between them.
	 *
	 * @author Michael W. Floyd
	 * @version 0.3
	 *
	 */
	@Override
	public float pairwiseDistance(SensoryItem f){
		//check the parameters
		if(f == null){
			throw new IllegalArgumentException("Null Spatial2DFeatures given to pairwiseDistance method");
		}
		//make sure they are of the correct type
		if( !(f instanceof Spatial2DObject)){
			throw new IllegalArgumentException("Features given to pairwiseDistance method must be Spatial2DObjects");
		}
		
		//if this distance calculation is not set, we call the method from the superclass
		if(Spatial2DObject.m_distanceCalculation == null){
			return super.pairwiseDistance(f);
		}
		
		return Spatial2DObject.m_distanceCalculation.pairwiseDistance(this, f);
	}
	
	/** Overrides the equals method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Override
	public boolean equals(Object o){
		//see if same object
		if(this == o){
			return true;
		}
		
		//check for null or wrong class
		if(o == null || !(o instanceof Spatial2DObject)){
			return false;
		}
		
		//typecast the object
		Spatial2DObject s2D = (Spatial2DObject)o;
		
		//make sure the super class attributes are the same
		if(!super.equals(s2D)){
			return false;
		}
		
		return ( (this.m_direction == s2D.getDirection()) && (this.m_distance == s2D.getDistance()) );
	}
}
