/** AgentInputsTest.java in the package tests.junit.org.JIFSA of the JIFSA project.
    Originally created 6-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.JIFSA.AgentInputs;
import org.JIFSA.SensoryItem;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/** Units tests for the org.JIFSA.AgentInputs class.
 * 
 * @author Michael W. Floyd
 * @since  0.1
 */
public class AgentInputsTest {
	
	/** Test to make sure the constructor does not
	 * cause any exceptions and that it is initiall
	 * empty
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void noargConstructor(){
		AgentInputs v = new AgentInputs();
		assertEquals(v.getNumberSensoryItems(),0);
	}
	
	/** Tests the copy constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void copyConstructor(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<50;ii++){
			SensoryItem f1 = new SensoryItem("Feature1");
			SensoryItem f2 = new SensoryItem("Feature2");
			SensoryItem f3 = new SensoryItem("Feature3");
			v.addSensoryItem(f1);
			v.addSensoryItem(f2);
			v.addSensoryItem(f3);
		}
		
		AgentInputs copy = new AgentInputs(v);
		
		assertEquals(v.getNumberSensoryItems(),copy.getNumberSensoryItems());
		assertEquals(v.getNumberSensoryItems("Feature1"),copy.getNumberSensoryItems("Feature1"));
		assertEquals(v.getNumberSensoryItems("Feature2"),copy.getNumberSensoryItems("Feature1"));
		assertEquals(v.getNumberSensoryItems("Feature3"),copy.getNumberSensoryItems("Feature1"));
	}
	
	/** Tests the copy constructor when a null parameter is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void copyConstructor_nullParam(){
		new AgentInputs(null);
	}
	
	/** Test to make sure the function properly
	 * responds to an attempt to add a null value
	 * for a feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void addFeature_nullParam(){
		AgentInputs v = new AgentInputs();
		v.addSensoryItem(null);
	}
	
	/** Tests adding a single feature to the AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void addFeature_addSingle(){
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("New SensoryItem");
		v.addSensoryItem(f);
		assertEquals(v.getNumberSensoryItems(),1);
		assertEquals(v.getNumberSensoryItems("New SensoryItem"),1);
	}
	
	/** Tests to make sure features of differnt types can
	 * be added.
	 *
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void addFeature_differentTypes(){
		AgentInputs v = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		SensoryItem f3 = new SensoryItem("Feature3");
		
		v.addSensoryItem(f1);
		v.addSensoryItem(f2);
		v.addSensoryItem(f3);
		
		assertEquals(v.getNumberSensoryItems(),3);
		assertEquals(v.getNumberSensoryItems("Feature1"),1);
		assertEquals(v.getNumberSensoryItems("Feature2"),1);
		assertEquals(v.getNumberSensoryItems("Feature3"),1);
		
	}
	
	/** Tests adding multiple features of the same type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void addFeature_multipleSameType(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<100;ii++){
			SensoryItem f = new SensoryItem("SensoryItem Name");
			v.addSensoryItem(f);
		}
		
		assertEquals(v.getNumberSensoryItems(),100);
		assertEquals(v.getNumberSensoryItems("SensoryItem Name"),100);
	}
	
	/** Test the removal of a feature type when it is the
	 * only type of feature that has been added.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void removeFeatureType_onlyOneType(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<50;ii++){
			SensoryItem f = new SensoryItem("SensoryItem Name");
			v.addSensoryItem(f);
		}
		
		assertEquals(v.getNumberSensoryItems(),50);
		assertEquals(v.getNumberSensoryItems("SensoryItem Name"),50);
		
		v.removeSensoryItemType("SensoryItem Name");
		
		assertEquals(v.getNumberSensoryItems(),0);
		assertEquals(v.getNumberSensoryItems("SensoryItem Name"),0);
	}
	
	/** Tests removing features of different types
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void removeFeatureType_multipleTypes(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<50;ii++){
			SensoryItem f1 = new SensoryItem("Feature1");
			SensoryItem f2 = new SensoryItem("Feature2");
			SensoryItem f3 = new SensoryItem("Feature3");
			v.addSensoryItem(f1);
			v.addSensoryItem(f2);
			v.addSensoryItem(f3);
		}
		
		assertEquals(v.getNumberSensoryItems(),150);
		assertEquals(v.getNumberSensoryItems("Feature1"),50);
		assertEquals(v.getNumberSensoryItems("Feature2"),50);
		assertEquals(v.getNumberSensoryItems("Feature3"),50);
		
		v.removeSensoryItemType("Feature1");
		
		assertEquals(v.getNumberSensoryItems(),100);
		assertEquals(v.getNumberSensoryItems("Feature1"),0);
		assertEquals(v.getNumberSensoryItems("Feature2"),50);
		assertEquals(v.getNumberSensoryItems("Feature3"),50);
		
		v.removeSensoryItemType("Feature2");
		
		assertEquals(v.getNumberSensoryItems(),50);
		assertEquals(v.getNumberSensoryItems("Feature1"),0);
		assertEquals(v.getNumberSensoryItems("Feature2"),0);
		assertEquals(v.getNumberSensoryItems("Feature3"),50);
		
		v.removeSensoryItemType("Feature3");
		
		assertEquals(v.getNumberSensoryItems(),0);
		assertEquals(v.getNumberSensoryItems("Feature1"),0);
		assertEquals(v.getNumberSensoryItems("Feature2"),0);
		assertEquals(v.getNumberSensoryItems("Feature3"),0);
	}
	
	/** Tests to make sure that an attempt to remove
	 * a feature that is not in the AgentInputs does not
	 * change the features in the AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void removeFeatures_doesNotExist(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<10;ii++){
			SensoryItem f = new SensoryItem("SensoryItem");
			v.addSensoryItem(f);
		}
		
		assertEquals(v.getNumberSensoryItems(),10);
		assertEquals(v.getNumberSensoryItems("SensoryItem"),10);
		
		v.removeSensoryItemType("Not Contained");
		
		assertEquals(v.getNumberSensoryItems(),10);
		assertEquals(v.getNumberSensoryItems("SensoryItem"),10);
	}
	
	/** Tests to make sure the removeFeatureType
	 * function responds as expected to a null
	 * feature name.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void removeFeatures_nullFeatureName(){
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("SensoryItem Name");
		v.addSensoryItem(f);
		v.removeSensoryItemType(null);
	}
	
	/** Test to make sure that the get features
	 * method returns the features of a given type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatures_singleType(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<10;ii++){
			SensoryItem f = new SensoryItem("SensoryItem");
			v.addSensoryItem(f);
		}
		
		assertEquals(v.getNumberSensoryItems(),10);
		assertEquals(v.getNumberSensoryItems("SensoryItem"),10);
		
		List<SensoryItem> l = v.getSensoryItems("SensoryItem");
		
		assertNotNull(l);
		assertEquals(l.size(),10);
		
		for(SensoryItem feat: l){
			assertTrue(feat.getFeatureName().equals("SensoryItem"));
		}
		
		assertEquals(v.getNumberSensoryItems(),10);
		assertEquals(v.getNumberSensoryItems("SensoryItem"),10);
	}
	
	/** Test to make sure that the get features
	 * method returns the features of a given type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatures_multipleType(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<10;ii++){
			SensoryItem f1 = new SensoryItem("Feature1");
			SensoryItem f2 = new SensoryItem("Feature2");
			SensoryItem f3 = new SensoryItem("Feature3");
			v.addSensoryItem(f1);
			v.addSensoryItem(f2);
			v.addSensoryItem(f3);
		}
		
		assertEquals(v.getNumberSensoryItems(),30);
		assertEquals(v.getNumberSensoryItems("Feature1"),10);
		assertEquals(v.getNumberSensoryItems("Feature2"),10);
		assertEquals(v.getNumberSensoryItems("Feature3"),10);
		
		List<SensoryItem> l1 = v.getSensoryItems("Feature1");
		List<SensoryItem> l2 = v.getSensoryItems("Feature2");
		List<SensoryItem> l3 = v.getSensoryItems("Feature3");
		
		assertNotNull(l1);
		assertNotNull(l2);
		assertNotNull(l3);
		
		assertEquals(l1.size(),10);
		assertEquals(l2.size(),10);
		assertEquals(l3.size(),10);
		
		for(SensoryItem feat: l1){
			assertTrue(feat.getFeatureName().equals("Feature1"));
		}
		for(SensoryItem feat: l2){
			assertTrue(feat.getFeatureName().equals("Feature2"));
		}
		for(SensoryItem feat: l3){
			assertTrue(feat.getFeatureName().equals("Feature3"));
		}
		
		assertEquals(v.getNumberSensoryItems(),30);
		assertEquals(v.getNumberSensoryItems("Feature1"),10);
		assertEquals(v.getNumberSensoryItems("Feature2"),10);
		assertEquals(v.getNumberSensoryItems("Feature3"),10);
	}
	
	/** Ensures the getFeatures function
	 * responds as expected when a null feature name
	 * is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getFeatures_nullFileName(){
		AgentInputs v = new AgentInputs();
		v.getSensoryItems(null);
	}
	
	/** Tests that a null value is returned
	 * if a feature type is not contained in
	 * the AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatures_notContained(){
		AgentInputs v = new AgentInputs();
		assertNull(v.getSensoryItems("Not Contained"));
	}
	
	
	/** Tests that TRUE is returned for doesContain
	 *  for an AgentInputs with a single feature type
	 * 
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void doesContain_singleType(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=1;ii<10;ii++){
			SensoryItem f = new SensoryItem("SensoryItem");
			v.addSensoryItem(f);
		}
		
		assertTrue(v.doesContain("SensoryItem"));
	}
	
	
	/** Tests that FALSE is returned for doesContain
	 *  for an AgentInputs when the feature has been
	 *  removed
	 * 
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void doesContain_removed(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=1;ii<10;ii++){
			SensoryItem f = new SensoryItem("SensoryItem");
			v.addSensoryItem(f);
		}
		
		assertTrue(v.doesContain("SensoryItem"));
		
		v.removeSensoryItemType("SensoryItem");
		
		assertFalse(v.doesContain("SensoryItem"));
	}
	
	/** Tests that TRUE is returned for doesContain
	 *  for an AgentInputs with multiple feature types.
	 *  It also checks that FALSE is returned after the
	 *  features are removed
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void doesContain_multipleTypesAndRemoval(){
		AgentInputs v = new AgentInputs();
		
		for(int ii=0;ii<50;ii++){
			SensoryItem f1 = new SensoryItem("Feature1");
			SensoryItem f2 = new SensoryItem("Feature2");
			SensoryItem f3 = new SensoryItem("Feature3");
			v.addSensoryItem(f1);
			v.addSensoryItem(f2);
			v.addSensoryItem(f3);
		}
		
		assertTrue(v.doesContain("Feature1"));
		assertTrue(v.doesContain("Feature2"));
		assertTrue(v.doesContain("Feature3"));
		
		v.removeSensoryItemType("Feature1");
		
		assertFalse(v.doesContain("Feature1"));
		assertTrue(v.doesContain("Feature2"));
		assertTrue(v.doesContain("Feature3"));
		
		v.removeSensoryItemType("Feature2");
		
		assertFalse(v.doesContain("Feature1"));
		assertFalse(v.doesContain("Feature2"));
		assertTrue(v.doesContain("Feature3"));
		
		v.removeSensoryItemType("Feature3");
		
		assertFalse(v.doesContain("Feature1"));
		assertFalse(v.doesContain("Feature2"));
		assertFalse(v.doesContain("Feature3"));
	}
	
	/** Makes sure the doesContain function returns
	 * false if a feature type is not in the AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void doesContain_notContained(){
		AgentInputs v = new AgentInputs();
		assertFalse(v.doesContain("Not Contained"));
	}
	
	/** Ensures the doesContain function
	 * responds as expected when a null feature name
	 * is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void doesContain_nullFileName(){
		AgentInputs v = new AgentInputs();
		v.doesContain(null);
	}
	
	/** Ensures the getNumberFeatures function
	 * responds as expected when a null feature name
	 * is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getNumberFeatures_nullFeatureName(){
		AgentInputs v = new AgentInputs();
		v.getNumberSensoryItems(null);
		
	}
	
	/** Test to see that if a feature is not in
	 * an AgentInputs that the number returned is
	 * zero.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getNumberFeatures_featureNotContained(){
		AgentInputs v = new AgentInputs();
		assertEquals(v.getNumberSensoryItems("New SensoryItem"),0);
	}
	
	/** Tests to see if an AgentInputs with no features
	 * returns a feature name list that is empty
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatureNames_none(){
		AgentInputs v = new AgentInputs();
		List<String> featureNames = v.getSensoryItemNames();
		assertEquals(featureNames.size(),0);
	}
	
	/** Tests to see if an AgentInputs with no features
	 * returns a feature name list that is empty after
	 * all features have been removed
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatureNames_noneAfterRemoval(){
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("SensoryItem");
			
		v.addSensoryItem(f);
		
		v.removeSensoryItemType("SensoryItem");
		
		List<String> featureNames = v.getSensoryItemNames();
		assertEquals(featureNames.size(),0);
	}
	
	/** Tests to see that if a feature of a certain type
	 * is added that the feature name appears in the 
	 * feature name list.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatureNames_oneType(){
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("SensoryItem");
			
		v.addSensoryItem(f);
		
		List<String> featureNames = v.getSensoryItemNames();
		assertEquals(featureNames.size(),1);
		assertTrue(featureNames.contains("SensoryItem"));
	}
	
	
	/** Tests to see that if features are added
	 * that their feature names appears in the 
	 * feature name list.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatureNames_severalTypes(){
		AgentInputs v = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		SensoryItem f3 = new SensoryItem("Feature3");
		
		v.addSensoryItem(f1);
		v.addSensoryItem(f2);
		v.addSensoryItem(f3);
		
		List<String> featureNames = v.getSensoryItemNames();
		assertEquals(featureNames.size(),3);
		assertTrue(featureNames.contains("Feature1"));
		assertTrue(featureNames.contains("Feature2"));
		assertTrue(featureNames.contains("Feature3"));
	}
	
	/** Tests to make sure the feature name list
	 * cannot modify the underlying data
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getFeatureNames_modify(){
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("SensoryItem");
			
		v.addSensoryItem(f);
		
		List<String> featureNames = v.getSensoryItemNames();
		featureNames.clear();
		featureNames = v.getSensoryItemNames();
		assertEquals(featureNames.size(),1);
		assertTrue(featureNames.contains("SensoryItem"));
	}
	
	
	/** Test equals with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_null(){
		AgentInputs v = new AgentInputs();
		assertFalse(v.equals(null));
	}
	
	/** Tests equals with a different object type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_differentClass(){
		AgentInputs v = new AgentInputs();
		String s = new String();
		assertFalse(v.equals(s));
	}
	
	/** Tests two default constructed objects
	 * for equality.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_empty(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		assertEquals(v1,v2);
	}
	
	/** Tests equality of two same AgentInputss
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_equalNotEmpty(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("SensoryItem");
		SensoryItem f12 = new SensoryItem("Feature1");
		SensoryItem f21 = new SensoryItem("SensoryItem");
		SensoryItem f22 = new SensoryItem("Feature1");
		v1.addSensoryItem(f11);
		v1.addSensoryItem(f12);
		v2.addSensoryItem(f21);
		v2.addSensoryItem(f22);
		
		assertEquals(v1,v2);
	}
	
	/** Tests equality of two same AgentInputss
	 * with different feature order
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_equalDifferentOrder(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("SensoryItem");
		SensoryItem f12 = new SensoryItem("Feature1");
		SensoryItem f21 = new SensoryItem("SensoryItem");
		SensoryItem f22 = new SensoryItem("Feature1");
		v1.addSensoryItem(f12);
		v1.addSensoryItem(f11);
		v2.addSensoryItem(f21);
		v2.addSensoryItem(f22);
		
		assertEquals(v1,v2);
	}
	
	/** Tests equality of two AgentInputss
	 * with different number of features
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_notEqualDifferentSize(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("SensoryItem");
		SensoryItem f21 = new SensoryItem("SensoryItem");
		SensoryItem f22 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f11);
		v2.addSensoryItem(f21);
		v2.addSensoryItem(f22);
		
		assertFalse(v1.equals(v2));
	}
	
	/** Tests equality of two AgentInputss
	 * with different number of features of each type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test 
	public void equals_notEqualDifferentFeatureDistribution(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("Feature1");
		SensoryItem f12 = new SensoryItem("Feature2");
		SensoryItem f13 = new SensoryItem("Feature2");
		SensoryItem f21 = new SensoryItem("Feature1");
		SensoryItem f22 = new SensoryItem("Feature1");
		SensoryItem f23 = new SensoryItem("Feature2");
		v1.addSensoryItem(f11);
		v1.addSensoryItem(f12);
		v1.addSensoryItem(f13);
		v2.addSensoryItem(f21);
		v2.addSensoryItem(f22);
		v2.addSensoryItem(f23);
		
		assertFalse(v1.equals(v2));
	}
	
	
	/** Tests equality of two AgentInputss
	 * that are different but have same number of
	 * features
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void equals_notEqualSameSize(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("SensoryItem");
		SensoryItem f12 = new SensoryItem("Feature2");
		SensoryItem f21 = new SensoryItem("SensoryItem");
		SensoryItem f22 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f11);
		v1.addSensoryItem(f12);
		v2.addSensoryItem(f21);
		v2.addSensoryItem(f22);
		
		assertFalse(v1.equals(v2));
	}
	
	/** Tests to make sure that an item is equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_self(){
		AgentInputs v1 = new AgentInputs();
		SensoryItem f11 = new SensoryItem("SensoryItem");
		SensoryItem f12 = new SensoryItem("Feature2");
		v1.addSensoryItem(f11);
		v1.addSensoryItem(f12);
		assertTrue(v1.equals(v1));
	}
	
	/** Tests to make sure that two AgentInputs are equal
	 * if they have the same feature types (and the same number)
	 * but the lists of those features do not contain the same features.
	 * This test needed to create an anonymous class of a SensoryItem
	 * so that a non-equal feature could be artificially be put into
	 * the other AgentInputs list. This is because the SensoryItem equals method only
	 * compares the name of the features to test equality , but subclasses
	 * of SensoryItem may use other parameters. Therefore features in the same list
	 * (with the same feature name) would always be equal and we could not properly
	 * test this condition without overriding the equals method.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_doesNotContainSameFeatures(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		
		SensoryItem f1 = new SensoryItem("SensoryItem"){
			private static final long serialVersionUID = 1L;

			@Override
			public boolean equals(Object o){
				return (this == o);
			}
		};
		SensoryItem f2 = new SensoryItem("SensoryItem");
		
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		assertFalse(v1.equals(v2));
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(AgentInputsTest.class);
	}
}
