/** AgentTest.java in the package tests.junit.org.JIFSA of the JIFSA project.
    Originally created 17-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.SensoryItem;
import org.JIFSA.preprocessing.filters.CaseFilter;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.ActionEstimation;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.GlobalDistanceMeasure;
import org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithm;
import org.JIFSA.reasoning.distance.penalty.ConstantPenalty;

import org.junit.Before;
import org.junit.Test;

/** Unit tests for the org.JIFSA.Agent class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class AgentTest {

	private CaseBase caseBase;
	private AgentAction expected1;
	private AgentAction expected2;
	private AgentInputs vision1;
	private AgentInputs vision2;

	private CaseBaseSearch cbSearch;
	private ActionSelection actionSelection;
	
	/** Create the CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createCaseBase(){
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("firstFeature");
		av1.addSensoryItem(f1);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		this.expected1 = new AgentAction("firstActionType");
		act1.add(this.expected1);
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f2 = new SensoryItem("secondFeature");
		av2.addSensoryItem(f2);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		AgentAction a2 = new AgentAction("secondActionType");
		act2.add(a2);
		Case c2 = new Case(av2,act2);
		
		AgentInputs av3 = new AgentInputs();
		SensoryItem f3 = new SensoryItem("firstFeature");
		SensoryItem f4 = new SensoryItem("secondFeature");
		av3.addSensoryItem(f3);
		av3.addSensoryItem(f4);
		List<AgentAction> act3 = new ArrayList<AgentAction>();
		this.expected2 = new AgentAction("thirdActionType");
		act3.add(this.expected2);
		Case c3 = new Case(av3,act3);
		
		this.vision1 = new AgentInputs();
		SensoryItem f5 = new SensoryItem("firstFeature");
		this.vision1.addSensoryItem(f5);

		
		this.vision2 = new AgentInputs();
		SensoryItem f6 = new SensoryItem("firstFeature");
		SensoryItem f7 = new SensoryItem("secondFeature");
		this.vision2.addSensoryItem(f6);
		this.vision2.addSensoryItem(f7);

		
		this.caseBase = new CaseBase();
		this.caseBase.addCase(c1);
		this.caseBase.addCase(c2);
		this.caseBase.addCase(c3);
	}
	
	
	/** Create the CaseBaseSearch and sets the algorithms 
	 * that are used to calculate distance/penalties
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createCloseCaseSearchAndAlgorithms(){
		this.cbSearch = new NearestNeighbourSearch(1);
		
		SensoryItem.setDistanceCalculation(new EqualityDistanceAlgorithm());
		SensoryItem.setPenaltyDistanceCalculation(new ConstantPenalty(1000));
			
		Weights w = new Weights(1.0f);
		GlobalDistanceMeasure gd = new OrderIndexMatchingAlgorithm(w);
		Case.setGlobalDistanceCalculation(gd);
	}
	
	/** Create the ActionSelection
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createActionSelection(){
		ActionEstimation ae = new LastActionEstimate();
		this.actionSelection = new ClosestNeighbourSelection(ae); 
	}
	
	/** Tests the constructor with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullAll(){
		new Agent(null,null,null);
	}
	
	/** Tests the constructor with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullCaseBase(){
		new Agent(null,this.cbSearch,this.actionSelection);
	}
	
	/** Tests the constructor with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullCloseCaseSearch(){
		new Agent(this.caseBase,null,this.actionSelection);
	}
	
	/** Tests the constructor with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullActionSelection(){
		new Agent(this.caseBase,this.cbSearch,null);
	}
	
	/** Test to make sure the senseEnvironment method
	 * does not return null when given an empty
	 * AgentInputs.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void senseEnvironment_notreturnnull(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		AgentInputs av = new AgentInputs();
		assertNotNull(a.senseEnvironment(av));
	}

	
	/** Tests the method using valid parameters.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_valid(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		AgentAction action1 = a.senseEnvironment(this.vision1);
		assertEquals(action1, this.expected1);
		AgentAction action2 = a.senseEnvironment(this.vision2);
		assertEquals(action2, this.expected2);
	}
	
	/** Test to make sure the senseEnvironment method
	 * throws an exception when given a null
	 * AgentInputs.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_nullCase(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		a.senseEnvironment(null);
	}
	
	/** Tests the method with an empty CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_emptyCaseBase(){
		CaseBase empty  = new CaseBase();
		new Agent(empty,this.cbSearch,this.actionSelection);
	}
	
	/** Tests when one of the supplied filters is null
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setFilters_nullFilter(){
		CaseFilter[] filters = new CaseFilter[3];
		
		CaseFilter validFilter = new CaseFilter(){

			public Case filter(Case initialCase) {
				AgentInputs ai = initialCase.getInputs();
				ai.addSensoryItem(new SensoryItem("secondFeature"));
				List<AgentAction> actions = initialCase.getActions();
				return new Case(ai,actions);
			}
			
		};
		
		filters[0] = validFilter;
		filters[1] = null;
		filters[2] = validFilter;
		
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		a.setFilters(filters);		
	}
	
	/** Tests when a single filter is applied
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_withOneFilter(){
		CaseFilter[] filters = new CaseFilter[1];
		
		//add a simple filter that adds a SensoryItem
		filters[0] = new CaseFilter(){

			public Case filter(Case initialCase) {
				AgentInputs ai = initialCase.getInputs();
				ai.addSensoryItem(new SensoryItem("secondFeature"));
				List<AgentAction> actions = initialCase.getActions();
				return new Case(ai,actions);
			}
			
		};
		
	
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		a.setFilters(filters);
		AgentAction action1 = a.senseEnvironment(this.vision1);
		assertEquals(action1.getActionName(), "thirdActionType");
		AgentAction action2 = a.senseEnvironment(this.vision2);
		assertEquals(action2.getActionName(), "thirdActionType");
	}
	
	/** Tests when a multiple filters are applied
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_withMultipleFilters(){
		CaseFilter[] filters = new CaseFilter[2];
		
		//add a simple filter that clears all Features
		filters[0] = new CaseFilter(){

			public Case filter(Case initialCase) {
				AgentInputs ai = new AgentInputs();
				List<AgentAction> actions = initialCase.getActions();
				return new Case(ai,actions);
			}
			
		};
		
		//add a simple filter that adds a SensoryItem
		filters[1] = new CaseFilter(){

			public Case filter(Case initialCase) {
				AgentInputs ai = initialCase.getInputs();
				ai.addSensoryItem(new SensoryItem("secondFeature"));
				List<AgentAction> actions = initialCase.getActions();
				return new Case(ai,actions);
			}
			
		};
		
	
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		a.setFilters(filters);
		AgentAction action1 = a.senseEnvironment(this.vision1);
		assertEquals(action1.getActionName(), "secondActionType");
		AgentAction action2 = a.senseEnvironment(this.vision2);
		assertEquals(action2.getActionName(), "secondActionType");
	}
	
	/** Tests when a null value is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setCaseBase_null(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		a.setCaseBase(null);
	}
	
	/** Tests when an invalid value is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setCaseBase_invalidCB(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		CaseBase cb = new CaseBase();
		a.setCaseBase(cb);
	}
	
	/** Tests when an valid value is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void setCaseBase_valid(){
		Agent a = new Agent(this.caseBase,this.cbSearch,this.actionSelection);
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> al1 = new ArrayList<AgentAction>();
		al1.add(new AgentAction("anAction"));
		Case c1 = new Case(ai,al1);
		List<AgentAction> al2 = new ArrayList<AgentAction>();
		al2.add(new AgentAction("secondAction"));
		Case c2 = new Case(ai,al2);
		
		CaseBase cb1 = new CaseBase();
		cb1.addCase(c1);
		
		CaseBase cb2 = new CaseBase();
		cb2.addCase(c2);
		
		a.setCaseBase(cb1);
		AgentAction res1 = a.senseEnvironment(ai);
		a.setCaseBase(cb2);
		AgentAction res2 = a.senseEnvironment(ai);
		
		assertEquals(res1.getActionName(), "anAction");
		assertEquals(res2.getActionName(), "secondAction");	
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(AgentTest.class);
	}
}
