/** CaseBaseTest.java in the package tests.junit.org.JIFSA of the JIFSA project.
    Originally created 18-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.CaseBase;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.junit.Test;

/** Unit tests for the org.JIFSA.CaseBase class
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class CaseBaseTest {


	/** Test to make sure the CaseBase has
	 * an initial size of zero.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void initialSizeIsZero(){
		CaseBase c = new CaseBase();
		//make sure the case base size is zero
		assertEquals(c.getCasebaseSize(),0);
		//make sure the list of cases is empty
		List<Case> l = c.getCaseList();
		assertEquals(l.size(), 0);
	}
	
	/** Test the copy constructor when a null
	 * parameter is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void copyConstructor_nullParam(){
		new CaseBase(null);
	}
	
	/** Tests the copy constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void copyConstructor(){
		Case c1 = new Case();
		
		AgentInputs v = new AgentInputs();
		SensoryItem f =  new SensoryItem("SensoryItem");
		v.addSensoryItem(f);
		List<AgentAction> l = new ArrayList<AgentAction>();
		AgentAction a1 = new AgentAction("Action1");
		AgentAction a2 = new AgentAction("Action2");
		l.add(a1);
		l.add(a2);
		Case c2 = new Case(v,l);
		
		CaseBase cb1 = new CaseBase();
		cb1.addCase(c1);
		cb1.addCase(c2);
		
		CaseBase copy = new CaseBase(cb1);
		assertTrue(cb1.equals(copy));
		
	}
	/** Test to make sure the a null Case
	 * cannot be added to the CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void addCase_nullCase(){
		CaseBase c = new CaseBase();
		c.addCase(null);
	}
	
	/** Test to make sure the a getList() function
	 * does not return a null CaseBase when empty
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getList_notNull(){
		CaseBase c = new CaseBase();
		assertNotNull(c.getCaseList());
	}
	
	/** Test to make sure the addCase method
	 * increases the size of the CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void addCase_ensureSizeInc(){
		CaseBase c = new CaseBase();
		for(int ii=1;ii<1000;ii++){
			Case sc = new Case();
			c.addCase(sc);
			assertEquals(c.getCasebaseSize(), ii);
		}
		
	}
	
	/** Test to make sure the addCase method
	 * properly adds to the CaseBase list
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void addCase_ensureProperAdd(){
		CaseBase c = new CaseBase();
		for(int ii=0;ii<1000;ii++){
			Case sc = new Case();
			c.addCase(sc);
			List<Case> l = c.getCaseList();
			assertEquals(l.size(),c.getCasebaseSize());
			assertEquals(l.get(ii),sc);
		}
	}
	
	/** Test to make sure we do not return the actual
	 * list, but rather a copy so it cannot be
	 * modified
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getList_notModifiable(){
		CaseBase c = new CaseBase();
		for(int ii=0;ii<1000;ii++){
			Case sc = new Case();
			c.addCase(sc);
		}
		
		List<Case> l = c.getCaseList();
		l.clear();
		l = c.getCaseList();
		assertEquals(l.size(),1000);
	}
	
	/** Tests to see if CaseBases of different
	 * sizes are found to be unequal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_unequalDifferentSizes(){
		CaseBase cb1 = new CaseBase();
		CaseBase cb2 = new CaseBase();
		Case c = new Case();
		cb2.addCase(c);
		
		assertFalse(cb1.equals(cb2));
	}
	
	/** See if two CaseBases of the same size
	 * that are different are found to be unequal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_unequalSameSize(){
		CaseBase cb1 = new CaseBase();
		CaseBase cb2 = new CaseBase();
		Case c1 = new Case();
		cb1.addCase(c1);
		
		AgentInputs v = new AgentInputs();
		SensoryItem f = new SensoryItem("SensoryItem");
		v.addSensoryItem(f);
		List<AgentAction> l = new ArrayList<AgentAction>();
		Case c2 = new Case(v,l);
		cb2.addCase(c2);
		
		assertFalse(cb1.equals(cb2));
	}
	
	/** Tests to ensure that the equals method
	 * returns false when a null value given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_null(){
		CaseBase cb = new CaseBase();
		assertFalse(cb.equals(null));
	}
	
	/** Tests to make sure two empty case bases are
	 * found to be equal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_equalEmpty(){
		CaseBase cb1 = new CaseBase();
		CaseBase cb2 = new CaseBase();
		assertEquals(cb1,cb2);
	}
	
	/** Tests to make sure two equal case bases are
	 * found to be equal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_equal(){
		CaseBase cb1 = new CaseBase();
		Case c11 = new Case();
		cb1.addCase(c11);
		
		AgentInputs v1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f1);
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		Case c12 = new Case(v1,l1);
		cb1.addCase(c12);
		
		CaseBase cb2 = new CaseBase();
		Case c21 = new Case();
		cb2.addCase(c21);
		
		AgentInputs v2 = new AgentInputs();
		SensoryItem f2 = new SensoryItem("SensoryItem");
		v2.addSensoryItem(f2);
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		Case c22 = new Case(v2,l2);
		cb2.addCase(c22);
		
		assertEquals(cb1,cb2);
	}
	
	/** Tests to make sure a CaseBase is
	 * equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_same(){
		CaseBase cb = new CaseBase();
		assertEquals(cb,cb);
	}
	
	/** See if two CaseBases that contain the same
	 * Cases but in different order are found to be
	 * equal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_differentOrder(){
		CaseBase cb1 = new CaseBase();
		Case c11 = new Case();
		cb1.addCase(c11);
		
		AgentInputs v1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f1);
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		Case c12 = new Case(v1,l1);
		cb1.addCase(c12);
		
		CaseBase cb2 = new CaseBase();
		Case c22 = new Case();
		
		AgentInputs v2 = new AgentInputs();
		SensoryItem f2 = new SensoryItem("SensoryItem");
		v2.addSensoryItem(f2);
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		Case c21 = new Case(v2,l2);
		cb2.addCase(c21);
		cb2.addCase(c22);
		
		assertEquals(cb1,cb2);
	}
	
	/** Test to make sure objects of different
	 * classes are said to be unequal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void testEquals_differntClass(){
		CaseBase cb = new CaseBase();
		String s = "String";
		assertFalse(cb.equals(s));
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(CaseBaseTest.class);
	}
}
