/** CaseTest.java in the package tests.junit.org.JIFSA of the JIFSA project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.distance.GlobalDistanceMeasure;
import org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithm;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/** Tests for the org.JIFSA.Case class
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class CaseTest {

	/** Test to see if the default constructor creates
	 * a Case with an AgentInputs that has no features
	 * and an empty AgentAction list
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void defaultConstructor(){
		Case c = new Case();
		AgentInputs v = c.getInputs();
		List<AgentAction> l = c.getActions();
		
		assertNotNull(v);
		assertNotNull(l);
		
		assertEquals(v.getNumberSensoryItems(),0);
		assertEquals(l.size(),0);
	}
	
	/** Tests if the constructor properly handles a null
	 * AgentInputs parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void agentVisionAgentActionConstructor_nullVision(){
		List<AgentAction> actions = new ArrayList<AgentAction>();
		
		new Case(null,actions);
	}
	
	/** Tests if the constructor properly handles a null
	 * AgentAction list parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void agentVisionAgentActionConstructor_nullAction(){
		AgentInputs v = new AgentInputs();
		
		new Case(v,null);
	}
	
	/** Test to see if the constructor creates
	 * a Case with an AgentInputs and AgentAction
	 * that are specified
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void agentVisionAgentActionConstructor(){
		List<AgentAction> actions = new ArrayList<AgentAction>();
		for(int ii=0;ii<10;ii++){
			AgentAction a = new AgentAction();
			actions.add(a);
		}
		AgentInputs vision = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		SensoryItem f3 = new SensoryItem("Feature1");
		vision.addSensoryItem(f1);
		vision.addSensoryItem(f2);
		vision.addSensoryItem(f3);
		
		Case c = new Case(vision,actions);
		AgentInputs newVision = c.getInputs();
		List<AgentAction> newActions = c.getActions();
		
		assertEquals(newVision.getNumberSensoryItems(),3);
		assertEquals(newVision.getNumberSensoryItems("Feature1"),2);
		assertEquals(newVision.getNumberSensoryItems("Feature2"),1);
		assertEquals(newActions.size(),10);	
	}
	
	/** Tests to make sure the AgentAction list that
	 * is returned cannot be used to modify the case
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void getActions_modify(){
		List<AgentAction> actions = new ArrayList<AgentAction>();
		AgentAction a = new AgentAction();
		actions.add(a);
		
		AgentInputs v = new AgentInputs();
		Case c = new Case(v,actions);
		
		List<AgentAction> caseActions = c.getActions();
		assertEquals(caseActions.size(),1);
		
		caseActions.clear();
		caseActions = c.getActions();
		assertEquals(caseActions.size(),1);
	}
	
	
	/** Test to make sure a Case is found to not
	 * be equal to null
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_null(){
		Case c = new Case();
		assertFalse(c.equals(null));
	}
	
	/** Test to make sure a Case is found to not
	 * be equal to a different Object class
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_differentClass(){
		Case c = new Case();
		String s = "String";
		assertFalse(c.equals(s));
	}
	
	/** Tests Cases with different AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_notEqualDiffentVision(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		AgentAction aa1 = new AgentAction("Action");
		AgentAction aa2 = new AgentAction("Action");
		l1.add(aa1);
		l2.add(aa2);
		
		Case c1 = new Case(v1,l1);
		Case c2 = new Case(v2,l2);
		
		assertFalse(c1.equals(c2));
	}
	
	/** Tests Cases with different AgentActions
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_notEqualDifferentActions(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("SensoryItem");
		SensoryItem f2 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		AgentAction aa1 = new AgentAction("Action1");
		AgentAction aa2 = new AgentAction("Action2");
		l1.add(aa1);
		l2.add(aa2);
		
		Case c1 = new Case(v1,l1);
		Case c2 = new Case(v2,l2);
		
		assertFalse(c1.equals(c2));
	}
	
	/** Tests Cases with same Actions in a different order
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_notEqualSameActionsWrongOrder(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		AgentAction aa11 = new AgentAction("Action1");
		AgentAction aa12 = new AgentAction("Action2");
		AgentAction aa21 = new AgentAction("Action2");
		AgentAction aa22 = new AgentAction("Action1");
		l1.add(aa11);
		l1.add(aa12);
		l2.add(aa21);
		l2.add(aa22);
		
		Case c1 = new Case(v1,l1);
		Case c2 = new Case(v2,l2);
		
		assertFalse(c1.equals(c2));
	}
	
	/** Tests Cases with different AgentActions and AgentInputs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_notEqualBothDifferent(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		AgentAction aa1 = new AgentAction("Action1");
		AgentAction aa2 = new AgentAction("Action2");
		l1.add(aa1);
		l2.add(aa2);
		
		Case c1 = new Case(v1,l1);
		Case c2 = new Case(v2,l2);
		
		assertFalse(c1.equals(c2));
	}
	
	/** Tests equality of Cases made with default constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_equalDefault(){
		Case c1 = new Case();
		Case c2 = new Case();
		assertEquals(c1,c2);
	}
	
	/** Tests equality of Cases not made with default constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_equalNotDefault(){
		AgentInputs v1 = new AgentInputs();
		AgentInputs v2 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("SensoryItem");
		SensoryItem f2 = new SensoryItem("SensoryItem");
		v1.addSensoryItem(f1);
		v2.addSensoryItem(f2);
		
		List<AgentAction> l1 = new ArrayList<AgentAction>();
		List<AgentAction> l2 = new ArrayList<AgentAction>();
		AgentAction aa1 = new AgentAction();
		AgentAction aa2 = new AgentAction();
		l1.add(aa1);
		l2.add(aa2);
		
		Case c1 = new Case(v1,l1);
		Case c2 = new Case(v2,l2);
		
		assertEquals(c1,c2);	
	}
	
	/** Makes sure it is equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_self(){
		Case c = new Case();
		assertEquals(c,c);
	}
	
	/** Test given the method a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setGlobalDistanceCalculation_null(){
		Case.setGlobalDistanceCalculation(null);
	}
	
	
	/** Tests the method when the algorithm
	 * has not been set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalStateException.class)
	public void pairwiseDistance_algorithmNotSet(){
		Case c1 = new Case();
		Case c2 = new Case();
		Case.pairwiseDistance(c1, c2);
	}
	
	/** Tests the method with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_bothNull(){
		Weights w = new Weights(1.0f);
		GlobalDistanceMeasure gdm = new GlobalDistanceMeasure(w){

			@Override
			public float pairwiseDistance(Case c1, Case c2) {
				//this method is only here so that we can create a GlobalDistanceMeasure object
				return 0;
			}
			
		};
		
		Case.setGlobalDistanceCalculation(gdm);
		Case.pairwiseDistance(null, null);
	}
	
	/** Tests the method with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_firstNull(){
		Weights w = new Weights(1.0f);
		GlobalDistanceMeasure gdm = new GlobalDistanceMeasure(w){

			@Override
			public float pairwiseDistance(Case c1, Case c2) {
				//this method is only here so that we can create a GlobalDistanceMeasure object
				return 0;
			}
			
		};
		
		Case c = new Case();
		
		Case.setGlobalDistanceCalculation(gdm);
		Case.pairwiseDistance(null, c);
	}
	
	
	/** Tests the method with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_secondNull(){
		Weights w = new Weights(1.0f);
		GlobalDistanceMeasure gdm = new GlobalDistanceMeasure(w){

			@Override
			public float pairwiseDistance(Case c1, Case c2) {
				//this method is only here so that we can create a GlobalDistanceMeasure object
				return 0;
			}
			
		};
		
		Case c = new Case();
		
		Case.setGlobalDistanceCalculation(gdm);
		Case.pairwiseDistance(c, null);
	}
	
	/** Tests the method with valid parameters
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_valid(){
		Weights w = new Weights(1.0f);
		GlobalDistanceMeasure gdm = new GlobalDistanceMeasure(w){

			@Override
			public float pairwiseDistance(Case c1, Case c2) {
				//set to an arbitrary value
				return 1384.89f;
			}
			
		};
		
		Case c = new Case();
		Case c2 = new Case();
		
		Case.setGlobalDistanceCalculation(gdm);
		assertEquals(Case.pairwiseDistance(c, c2), 1384.89f);
	}
	
	/** Test given the method on a valid parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void setGlobalDistanceCalculation_valid(){

		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		Case.setGlobalDistanceCalculation(oima);
		
		assertNotNull(Case.getGloablDistanceCalculation());
		assertTrue(Case.getGloablDistanceCalculation() instanceof OrderIndexMatchingAlgorithm);
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(CaseTest.class);
	}
	
	
}
