/** FeatureTest.java in the package tests.junit.org.JIFSA of the JIFSA project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.penalty.ConstantPenalty;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/** Tests for org.JIFSA.SensoryItem class
 * 
 * @author Michael W. Floyd
 * @since 0.1
 */
public class FeatureTest {

	/** Test the constructor to make sure
	 * the given feature name is used.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test 
	public void stringConstructor(){
		SensoryItem f = new SensoryItem("myFeature");
		String name = f.getFeatureName();
		assertTrue(name.equals("myFeature"));
	}
	
	/** Make sure the constructor deals as 
	 * expected with a null name
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test(expected=IllegalArgumentException.class)
	public void stringConstructor_nullParam(){
		new SensoryItem(null);
	}
	
	/** Tests equals with null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_null(){
		SensoryItem f = new SensoryItem("SensoryItem");
		assertFalse(f.equals(null));
	}
	
	/** Tests equals with different object type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_differentClass(){
		SensoryItem f = new SensoryItem("SensoryItem");
		String s = new String();
		assertFalse(f.equals(s));
	}
	
	/** Tests equals with same feature name
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_sameName(){
		SensoryItem f1 = new SensoryItem("SensoryItem");
		SensoryItem f2 = new SensoryItem("SensoryItem");
		assertEquals(f1,f2);
	}
	
	/** Tests equals with different feature name
	 * 
	 * @author Michael W. Floyd
	 * @since 0.1
	 */
	@Test
	public void equals_differentName(){
		SensoryItem f1 = new SensoryItem("Feature1");
		SensoryItem f2 = new SensoryItem("Feature2");
		assertFalse(f1.equals(f2));
	}
	
	/** Tests to make sure that an item is equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_self(){
		SensoryItem f1 = new SensoryItem("me");
		assertTrue(f1.equals(f1));
	}
	
	/** Test giving a null parameter as the distance
	 * calculation algorithm.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setDistanceCalculation_null(){
		SensoryItem.setDistanceCalculation(null);
	}
	
	/** Test giving a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullFirst(){
		SensoryItem f = new SensoryItem("feature");
		
		f.pairwiseDistance(null);
	}
	

	/** If the distance calculation algorithm has not
	 * been set an exception should be thrown.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalStateException.class)
	public void pairwiseDistance_distanceCalculationNotSet(){
		SensoryItem f1 = new SensoryItem("feature");
		SensoryItem f2 = new SensoryItem("feature");
		
		SensoryItem.removeDistanceCalculation();
		f1.pairwiseDistance(f2);
	}
	
	/** With the proper settings, the correct distance
	 * should be calculated.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test
	public void pairwiseDistance_validSettings(){
		EqualityDistanceAlgorithm adm = new EqualityDistanceAlgorithm();
		SensoryItem.setDistanceCalculation(adm);
		
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		SensoryItem f3 = new SensoryItem("yourFeature");
		
		assertEquals(f1.pairwiseDistance(f2), EqualityDistanceAlgorithm.MINDISTANCE);
		assertEquals(f1.pairwiseDistance(f3), EqualityDistanceAlgorithm.MAXDISTANCE);
		assertEquals(f2.pairwiseDistance(f3), EqualityDistanceAlgorithm.MAXDISTANCE);
	}
	
	/** Test giving a null parameter as the penalty
	 * calculation algorithm.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setPenaltyDistanceCalculation_null(){
		SensoryItem.setPenaltyDistanceCalculation(null);
	}
	

	
	/** If the penalty calculation algorithm has not
	 * been set an exception should be thrown.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalStateException.class)
	public void penaltyDistance_penaltyCalculationNotSet(){
		SensoryItem f1 = new SensoryItem("feature");
		
		SensoryItem.removePenaltyDistanceCalculation();
		f1.penaltyDistance();
	}
	
	/** With the proper settings, the correct penalty
	 * should be calculated.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test
	public void penaltyDistance_validSettings(){
		ConstantPenalty cp1 = new ConstantPenalty(0.0f);
		ConstantPenalty cp2 = new ConstantPenalty(-14f);
		ConstantPenalty cp3 = new ConstantPenalty(33.2f);
		
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("yourFeature");
		
		SensoryItem.setPenaltyDistanceCalculation(cp1);
		assertEquals(f1.penaltyDistance(), 0.0f);
		assertEquals(f2.penaltyDistance(), 0.0f);
		
		SensoryItem.setPenaltyDistanceCalculation(cp2);
		assertEquals(f1.penaltyDistance(), -14f);
		assertEquals(f2.penaltyDistance(), -14f);
		
		SensoryItem.setPenaltyDistanceCalculation(cp3);
		assertEquals(f1.penaltyDistance(), 33.2f);
		assertEquals(f2.penaltyDistance(), 33.2f);
		
	}
	
	/** Tests the XML generated for a SensoryItem
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 *
	 */
	@Test
	public void toXML(){
		SensoryItem item = new SensoryItem("myItem");
		String xml = item.toXML();
		String ideal = "<SensoryItem><FeatureName>myItem</FeatureName></SensoryItem>";
		assertEquals(xml, ideal);
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(FeatureTest.class);
	}
}
