/** ClassificationStatisticsWrapperTest.java in the package tests.junit.org.JIFSA.performance of the JIFSA project.
    Originally created 24-Nov-07

    Copyright (C) 2007 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.performance;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.performance.ClassificationStatisticsWrapper;
import org.JIFSA.performance.MeanExecutionTime;
import org.JIFSA.performance.PerformanceWrapper;
import org.JIFSA.performance.StatisticsWrapper;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.ActionEstimation;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.junit.Before;
import org.junit.Test;



/** Tests for the org.JIFSA.performance.ClassificationStatisticsWrapper class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class ClassificationStatisticsWrapperTest {

	private Agent ourAgent;
	private PerformanceWrapper ourWrapper;
	private ActionEstimation ourActionEstimation;
	
	/** Creates the Agent and PerformanceWrapper that
	 * will be wrapped around by the StatisticsWrapper.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createInternals(){
		CaseBase cb = new CaseBase();
		for(int ii=0;ii<20000;ii++){
			AgentInputs ai = new AgentInputs();
			List<AgentAction> aa = new ArrayList<AgentAction>();
			aa.add(new AgentAction("someAction"));
			Case c1 = new Case(ai, aa);
			cb.addCase(c1);
		}
		
		CaseBaseSearch cbSearch = new NearestNeighbourSearch(1);
		
		ActionSelection as = new ClosestNeighbourSelection(new LastActionEstimate());
		
		this.ourAgent = new Agent(cb, cbSearch, as);
		
		this.ourWrapper = new MeanExecutionTime(this.ourAgent);
		
		this.ourActionEstimation = new LastActionEstimate();
	}
	
	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor1_nullAE(){
		new ClassificationStatisticsWrapper(this.ourWrapper,null);
	}

	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor2_nullAE(){
		new ClassificationStatisticsWrapper(this.ourAgent,null);
	}
	
	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor1_nullWrapper(){
		PerformanceWrapper pw = null;
		new ClassificationStatisticsWrapper(pw,this.ourActionEstimation);
	}

	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor2_nullAgent(){
		Agent a = null;
		new ClassificationStatisticsWrapper(a,this.ourActionEstimation);
	}
	
	/** Make sure the statistics are properly generated
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void testvalidstats(){
		ClassificationStatisticsWrapper csw1 = new ClassificationStatisticsWrapper(this.ourAgent, this.ourActionEstimation);
		ClassificationStatisticsWrapper csw2 = new ClassificationStatisticsWrapper(this.ourWrapper, this.ourActionEstimation);
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> al1 = new ArrayList<AgentAction>();
		List<AgentAction> al2 = new ArrayList<AgentAction>();
		List<AgentAction> al3 = new ArrayList<AgentAction>();
		
		al1.add(new AgentAction("someAction"));
		al2.add(new AgentAction("anotherAction"));
		al3.add(new AgentAction("lastAction"));
		
		Case c1 = new Case(ai,al1);
		Case c2 = new Case(ai,al2);
		Case c3 = new Case(ai,al3);
		
		csw1.senseEnvironment(c1);
		csw1.senseEnvironment(c1);
		csw1.senseEnvironment(c1);
		csw1.senseEnvironment(c2);
		csw1.senseEnvironment(c3);
		
		csw2.senseEnvironment(c1);
		csw2.senseEnvironment(c1);
		csw2.senseEnvironment(c1);
		csw2.senseEnvironment(c2);
		csw2.senseEnvironment(c3);
		
		assertEquals(csw1.getClassificationAccuracy(), 0.6f);
		assertEquals(csw1.getPrecision("someAction"), 0.6f);
		assertEquals(csw1.getPrecision("anotherAction"), 0.0f);
		assertEquals(csw1.getPrecision("lastAction"), 0.0f);
		assertEquals(csw1.getRecall("someAction"), 1.0f);
		assertEquals(csw1.getRecall("anotherAction"), 0.0f);
		assertEquals(csw1.getRecall("lastAction"), 0.0f);
		assertEquals(csw1.getF1("someAction"), 0.75f);
		assertEquals(csw1.getF1("anotherAction"), 0.0f);
		assertEquals(csw1.getF1("lastAction"), 0.0f);
		assertEquals(csw1.getGlobalF1(), 0.25f);
		
		assertEquals(csw2.getClassificationAccuracy(), 0.6f);
		assertEquals(csw2.getPrecision("someAction"), 0.6f);
		assertEquals(csw2.getPrecision("anotherAction"), 0.0f);
		assertEquals(csw2.getPrecision("lastAction"), 0.0f);
		assertEquals(csw2.getRecall("someAction"), 1.0f);
		assertEquals(csw2.getRecall("anotherAction"), 0.0f);
		assertEquals(csw2.getRecall("lastAction"), 0.0f);
		assertEquals(csw2.getF1("someAction"), 0.75f);
		assertEquals(csw2.getF1("anotherAction"), 0.0f);
		assertEquals(csw2.getF1("lastAction"), 0.0f);
		assertEquals(csw2.getGlobalF1(), 0.25f);
		
		List<String> acts1 = csw1.getAllExpectedActions();
		List<String> acts2 = csw2.getAllExpectedActions();
		assertEquals(acts1.size(), 3);
		assertTrue(acts1.contains("someAction"));
		assertTrue(acts1.contains("anotherAction"));
		assertTrue(acts1.contains("lastAction"));
		assertEquals(acts2.size(), 3);
		assertTrue(acts2.contains("someAction"));
		assertTrue(acts2.contains("anotherAction"));
		assertTrue(acts2.contains("lastAction"));	
	}
	
	/** Test when the input Case has no actions associated
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void test_noactioninput(){
		ClassificationStatisticsWrapper csw = new ClassificationStatisticsWrapper(this.ourAgent, this.ourActionEstimation);
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		Case c = new Case(ai,act);
		
		csw.senseEnvironment(c);
		
		List<String> theactions = csw.getAllExpectedActions();
		assertEquals(theactions.size(), 1);
		assertTrue(theactions.contains(StatisticsWrapper.NOACTION));
	}
	
	/** Test when the input Case has no actions associated
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void test_noactionoutput(){
		CaseBase cb = new CaseBase();
		for(int ii=0;ii<20000;ii++){
			AgentInputs ai = new AgentInputs();
			List<AgentAction> aa = new ArrayList<AgentAction>();
			Case c1 = new Case(ai, aa);
			cb.addCase(c1);
		}
		
		CaseBaseSearch cbSearch = new NearestNeighbourSearch(1);
		
		ActionSelection as = new ClosestNeighbourSelection(new LastActionEstimate());
		
		Agent ag = new Agent(cb, cbSearch, as);
		ClassificationStatisticsWrapper csw = new ClassificationStatisticsWrapper(ag, this.ourActionEstimation);
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		act.add(new AgentAction("anAction"));
		Case c = new Case(ai,act);
		
		csw.senseEnvironment(c);
		
		List<String> theactions = csw.getAllExpectedActions();
		assertEquals(theactions.size(), 1);
		assertTrue(theactions.contains("anAction"));
	}
}
