/** LogWrapperTest.java in the package tests.junit.org.JIFSA.performance of the JIFSA project.
    Originally created 09-Aug-08

    Copyright (C) 2008 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.performance;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.performance.LogWrapper;
import org.JIFSA.performance.MeanExecutionTime;
import org.JIFSA.performance.PerformanceWrapper;
import org.JIFSA.performance.datatypes.CaseActionPair;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.junit.Before;
import org.junit.Test;

/** Tests for the org.JIFSA.performance.LogWrapper class
 * 
 * @author Michael W. Floyd
 * @since 0.4
 */
public class LogWrapperTest {

	private Agent ourAgent;
	private PerformanceWrapper ourWrapper;
	
	/** Creates the Agent and PerformanceWrapper that
	 * will be wrapped around by the LogWrapper.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Before
	public void createInternals(){
		CaseBase cb = new CaseBase();
		for(int ii=0;ii<20000;ii++){
			AgentInputs ai = new AgentInputs();
			List<AgentAction> aa = new ArrayList<AgentAction>();
			aa.add(new AgentAction("someAction"));
			Case c1 = new Case(ai, aa);
			cb.addCase(c1);
		}
		
		CaseBaseSearch cbSearch = new NearestNeighbourSearch(1);
		
		ActionSelection as = new ClosestNeighbourSelection(new LastActionEstimate());
		
		this.ourAgent = new Agent(cb, cbSearch, as);
		
		this.ourWrapper = new MeanExecutionTime(this.ourAgent);
	}
	
	/** Tests the constructor with a null Agent
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor1_nullAgent(){
		Agent ag = null;
		new LogWrapper(ag);
	}
	
	/** Tests the constructor with a null PerformanceWrapper
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor2_nullPerformanceWrapper(){
		PerformanceWrapper pw = null;
		new LogWrapper(pw);
	}
	
	/** Tests retrieving the pairs when there should be no
	 * logged pairs
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void getPairs_none(){
		//test with an agent
		LogWrapper lw = new LogWrapper(this.ourAgent);
		
		List<CaseActionPair> pairings = lw.getPairs();
		
		assertNotNull(pairings);
		assertEquals(pairings.size(), 0);
		
		//test with a performance wrapper
		LogWrapper lw2 = new LogWrapper(this.ourWrapper);
		
		List<CaseActionPair> pairings2 = lw2.getPairs();
		
		assertNotNull(pairings2);
		assertEquals(pairings2.size(), 0);
	}
	
	/** Tests retrieving the pairs when there should be some
	 * logged pair
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void getPairs_some(){
		Case c = new Case();
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> al = new ArrayList<AgentAction>();
		al.add(new AgentAction("someaction"));
		Case c2 = new Case(ai,al);
		
		
		//test with an agent
		LogWrapper lw = new LogWrapper(this.ourAgent);
		
		AgentAction act11 = lw.senseEnvironment(c);
		AgentAction act12 = lw.senseEnvironment(c2);
		
		List<CaseActionPair> pairings = lw.getPairs();
		
		assertNotNull(pairings);
		assertEquals(pairings.size(), 2);
		
		CaseActionPair p11 = pairings.get(0);
		assertNotNull(p11);
		assertEquals(c, p11.getCase());
		assertEquals(act11, p11.getAction());
		CaseActionPair p12 = pairings.get(1);
		assertNotNull(p12);
		assertEquals(c2, p12.getCase());
		assertEquals(act12, p12.getAction());
		
		//test with a performance wrapper
		LogWrapper lw2 = new LogWrapper(this.ourWrapper);
		
		AgentAction act21 = lw2.senseEnvironment(c);
		AgentAction act22 = lw2.senseEnvironment(c2);
		
		List<CaseActionPair> pairings2 = lw2.getPairs();
		
		assertNotNull(pairings2);
		assertEquals(pairings2.size(), 2);
		
		CaseActionPair p21 = pairings2.get(0);
		assertNotNull(p21);
		assertEquals(c, p21.getCase());
		assertEquals(act21, p21.getAction());
		CaseActionPair p22 = pairings2.get(1);
		assertNotNull(p22);
		assertEquals(c2, p22.getCase());
		assertEquals(act22, p22.getAction());
	}
	
	/** Creates an illegal state and makes sure we can handle
	 * that state.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalStateException.class)
	public void senseEnvironment_neitherSet(){
		
		//we create a subclass of LogWrapper so we can change
		// both Agent and PerformanceWrapper to null.
		LogWrapper lw= new LogWrapper(this.ourAgent){

			@Override
			public List<CaseActionPair> getPairs() {
				this.m_agent = null;
				this.m_internalwrapper = null;
				return null;
			}
			
		};
		
		//this should set them both null
		lw.getPairs();
		
		//now the exception should be thrown
		lw.senseEnvironment(new Case());
	}
	
	/** Tests when null values are given to senseEnvironment method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_null(){
		LogWrapper lw = new LogWrapper(this.ourAgent);
		
		lw.senseEnvironment(null);
	}
}
