/** MeanExecutionTimeTest.java in the package tests.junit.org.JIFSA.performance of the JIFSA project.
    Originally created 24-Nov-07

    Copyright (C) 2007 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.performance;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.performance.MeanExecutionTime;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.junit.Before;
import org.junit.Test;

/** Tests for the org.JIFSA.performance.MeanExecutionTime class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class MeanExecutionTimeTest {

	private Agent ourAgent;
	
	/** Create the agent that will be wrapped around
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createAgent(){
		CaseBase cb = new CaseBase();
		for(int ii=0;ii<20000;ii++){
			AgentInputs ai = new AgentInputs();
			List<AgentAction> aa = new ArrayList<AgentAction>();
			aa.add(new AgentAction("someAction"));
			Case c1 = new Case(ai, aa);
			cb.addCase(c1);
		}
		
		CaseBaseSearch cbSearch = new NearestNeighbourSearch(1);
		
		ActionSelection as = new ClosestNeighbourSelection(new LastActionEstimate());
		
		this.ourAgent = new Agent(cb, cbSearch, as);
	}
	
	/** Tests giving the constructor a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_null(){
		new MeanExecutionTime(null);
	}
	
	/** Tests that with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_null(){
		MeanExecutionTime met = new MeanExecutionTime(this.ourAgent);
		met.senseEnvironment(null);
	}
	
	/** Tests that the senseEnvironment method passes
	 * on the correct AgentAction.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_valid(){
		MeanExecutionTime met = new MeanExecutionTime(this.ourAgent);
		Case c = new Case();
		assertEquals(met.senseEnvironment(c),this.ourAgent.senseEnvironment(c.getInputs()));
	}
	
	/** Tests the method when no Cases have been
	 * sensed yet.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test
	public void getMeanExecutionTime_noSenses(){
		MeanExecutionTime met = new MeanExecutionTime(this.ourAgent);
		assertEquals(met.getMeanExecutionTime(), 0.0f);
	}
	
	/** Tests the method when one or more senses
	 * have been performed.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getMeanExecutionTime_someSenses(){
		MeanExecutionTime met = new MeanExecutionTime(this.ourAgent);
		for(int ii=0;ii<100;ii++){
			Case c = new Case();
			met.senseEnvironment(c);
		}
		
		float exec = met.getMeanExecutionTime();
		assertFalse(exec == 0.0f);
		//make sure it doesn't change
		assertEquals(exec, met.getMeanExecutionTime());
	}
}
