/** StatisticsBundleTest.java in the package tests.junit.org.JIFSA.performance of the JIFSA project.
    Originally created 02-Oct-08

    Copyright (C) 2008 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.performance;

import static org.junit.Assert.assertEquals;

import org.JIFSA.performance.StatisticsBundle;
import org.junit.Test;

/** Tests for the org.JIFSA.performance.StatisticsBundle class
 * 
 * @author Michael W. Floyd
 * @since 0.5
 */
public class StatisticsBundleTest {

	/** Tests when no labels are given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void contructor_nolabelsorstats(){
		float[] stats = new float[0];
		String[] labels = new String[0];
		new StatisticsBundle(stats, labels);
	}
	
	/** Tests when no stats are given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void contructor_onezero(){
		String[] labels = new String[10];
		float[] stats = new float[0];
		new StatisticsBundle(stats,labels);
	}
	
	/** Tests when the labels and stats are not of same length
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void contructor_unequal(){
		String[] labels = new String[10];
		float[] stats = new float[11];
		new StatisticsBundle(stats,labels);
	}
	
	/** Tests with valid stats and labels
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void constructor_valid(){
		float[] stats = new float[10];
		String[] labels = new String[10];
		for(int ii=0;ii<10;ii++){
			stats[ii] = ii;
			labels[ii] = "string" + ii;
		}
		
		StatisticsBundle sb = new StatisticsBundle(stats,labels);
		assertEquals(sb.getPrimaryLabel(), "string0");
		assertEquals(sb.getPrimaryStatistic(), 0);
		
		String[] allLabels = sb.getLabels();
		float[] allStats = sb.getAllStatistics();
		assertEquals(allLabels.length, allStats.length);
		
		for(int ii=0;ii<allLabels.length;ii++){
			assertEquals(allLabels[ii], "string" + ii);
			assertEquals(allStats[ii], ii);	
		}
	}
	
	/** Tests setting primary statistic when the index is too low
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setPrimaryStatistic_toolow(){
		float[] stats = new float[10];
		String[] labels = new String[10];
		for(int ii=0;ii<10;ii++){
			stats[ii] = ii;
			labels[ii] = "string" + ii;
		}
		
		StatisticsBundle sb = new StatisticsBundle(stats,labels);
		sb.setPrimaryStatistic(-1);
	}
	
	/** Tests setting primary statistic when the index is too high
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setPrimaryStatistic_toohigh(){
		float[] stats = new float[10];
		String[] labels = new String[10];
		for(int ii=0;ii<10;ii++){
			stats[ii] = ii;
			labels[ii] = "string" + ii;
		}
		
		StatisticsBundle sb = new StatisticsBundle(stats,labels);
		sb.setPrimaryStatistic(11);
	}
	
	/** Tests setting primary statistic when the index is valid
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void setPrimaryStatistic_valid(){
		float[] stats = new float[10];
		String[] labels = new String[10];
		for(int ii=0;ii<10;ii++){
			stats[ii] = ii;
			labels[ii] = "string" + ii;
		}
		
		StatisticsBundle sb = new StatisticsBundle(stats,labels);
		sb.setPrimaryStatistic(5);
		assertEquals(sb.getPrimaryStatistic(), 5);
		assertEquals(sb.getPrimaryLabel(),"string5");
		sb.setPrimaryStatistic(3);
		assertEquals(sb.getPrimaryStatistic(), 3);
		assertEquals(sb.getPrimaryLabel(),"string3");
		sb.setPrimaryStatistic(9);
		assertEquals(sb.getPrimaryStatistic(), 9);
		assertEquals(sb.getPrimaryLabel(),"string9");
	}
}
