/** StatisticsWrapperTest.java in the package tests.junit.org.JIFSA.performance of the JIFSA project.
    Originally created 24-Nov-07

    Copyright (C) 2007 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.performance;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.performance.MeanExecutionTime;
import org.JIFSA.performance.PerformanceWrapper;
import org.JIFSA.performance.StatisticsBundle;
import org.JIFSA.performance.StatisticsWrapper;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.ActionEstimation;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.junit.Before;
import org.junit.Test;


/** Tests for the org.JIFSA.performance.StatisticsWrapper class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class StatisticsWrapperTest {
	
	private Agent ourAgent;
	private PerformanceWrapper ourWrapper;
	private ActionEstimation ourActionEstimation;
	private CaseBase cb;
	private CaseBaseSearch cbSearch;
	private ActionSelection as;
	private StatisticsWrapper dummysw;
	
	/** Creates the Agent and PerformanceWrapper that
	 * will be wrapped around by the StatisticsWrapper.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void createInternals(){
		this.cb = new CaseBase();
		for(int ii=0;ii<20000;ii++){
			AgentInputs ai = new AgentInputs();
			List<AgentAction> aa = new ArrayList<AgentAction>();
			aa.add(new AgentAction("someAction"));
			Case c1 = new Case(ai, aa);
			this.cb.addCase(c1);
		}
		
		this.cbSearch = new NearestNeighbourSearch(1);
		
		this.as = new ClosestNeighbourSelection(new LastActionEstimate());
		
		this.ourAgent = new Agent(this.cb, this.cbSearch, this.as);
		
		this.ourWrapper = new MeanExecutionTime(this.ourAgent);
		
		this.ourActionEstimation = new LastActionEstimate();
		
		
		//the following are used for addPairHair
		PerformanceWrapper dummysimpleWrapper = new PerformanceWrapper(){

			//modify the presented Case slightly so we can identify it
			public AgentAction senseEnvironment(Case c) {
				//Do nothing useful, as it will never get called.
				return c.getActions().get(0);
			}
			
		};
		this.dummysw = new StatisticsWrapper(dummysimpleWrapper,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
	}
	

	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor1_nullAE(){
		new StatisticsWrapper(this.ourWrapper,null){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
	}


	/** Tests the constructor with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor2_nullAE(){
		new StatisticsWrapper(this.ourAgent,null){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
	}
	
	/** Tests the method with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_null(){
		StatisticsWrapper sw = new StatisticsWrapper(this.ourAgent,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
		
		sw.senseEnvironment(null);
	}
	
	/** Tests the method with a null parameter. Essential the
	 * same as above but uses the PerformanceWrapper instead of agent
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void senseEnvironment_null2(){
		StatisticsWrapper sw = new StatisticsWrapper(this.ourWrapper,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
		
		sw.senseEnvironment(null);
	}
	
	/** Makes sure the wrapper is invoked when it has been set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_wrapper(){
		
		PerformanceWrapper simpleWrapper = new PerformanceWrapper(){

			//modify the presented Case slightly so we can identify it
			public AgentAction senseEnvironment(Case c) {
				AgentAction initial = c.getActions().get(0);
				return new AgentAction(initial.getActionName() + "_edited");
			}
			
		};
		StatisticsWrapper sw = new StatisticsWrapper(simpleWrapper,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		act.add(new AgentAction("someName"));
		Case c = new Case(ai,act);
		AgentAction selected = sw.senseEnvironment(c);
		assertEquals(selected.getActionName(), "someName_edited");
	}
	
	/** Makes sure the agent is invoked when it has been set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void senseEnvironment_agent(){
		
		Agent simpleAgent = new Agent(this.cb,this.cbSearch,this.as){
			
			//return a known action
			@Override
			public AgentAction senseEnvironment(AgentInputs ai) {
				return new AgentAction("_modified");
			}
			
		};
		StatisticsWrapper sw = new StatisticsWrapper(simpleAgent,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				//Do nothing, implemented so we can create a StatisticsWrapper object		
			}
			
		};
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		act.add(new AgentAction("anotherName"));
		Case c = new Case(ai,act);
		AgentAction selected = sw.senseEnvironment(c);
		assertEquals(selected.getActionName(), "_modified");
	}
	
	/** Tests for when both agent and wrapper are set null
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalStateException.class)
	public void senseEnvironment_bothNull(){
		
		StatisticsWrapper sw = new StatisticsWrapper(this.ourAgent,this.ourActionEstimation){

			@Override
			protected void actionPair(AgentAction fromCase, AgentAction fromAgent) {
				this.m_agent = null;
			}
			
		};
		
		AgentInputs ai = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		act.add(new AgentAction("anotherName"));
		Case c = new Case(ai,act);
		//this one should set both action and wrapper null
		sw.senseEnvironment(c);
		sw.senseEnvironment(c);
	}
	
	/** Tests the method when no actions have been logged.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getClassificationAccuracy_empty(){
		assertEquals(this.dummysw.getClassificationAccuracy(),0.0f);
	}
	
	/** Tests the method when no action of one type have been logged.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getClassificationAccuracy_oneType() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "onlyAction", "onlyAction");
		addPair.invoke(this.dummysw, "onlyAction", "onlyAction");
		addPair.invoke(this.dummysw, "onlyAction", "onlyAction");
		addPair.invoke(this.dummysw, "onlyAction", "onlyAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 1.0f);
	}
	
	/** Tests the method when no action of several types have been 
	 * logged and all match perfectly.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getClassificationAccuracy_manyTypesPerfect() throws Exception{	
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "thirdAction", "thirdAction");
		addPair.invoke(this.dummysw, "thirdAction", "thirdAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 1.0f);
	}
	
	/** Tests the method when no action of several types have been 
	 * logged.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getClassificationAccuracy_manyTypes() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "thirdAction", "oneAction");
		addPair.invoke(this.dummysw, "thirdAction", "otherAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 0.4f);
	}
	
	/** Tests the method when it is called several times.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getClassificationAccuracy_severalTimes() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "thirdAction", "oneAction");
		addPair.invoke(this.dummysw, "thirdAction", "otherAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 0.4f);
		assertEquals(this.dummysw.getClassificationAccuracy(), 0.4f);
		
		addPair.invoke(this.dummysw, "oneAction", "thirdAction");
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "thirdAction", "thirdAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 0.375f);
		
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		
		assertEquals(this.dummysw.getClassificationAccuracy(), 0.5f);
	
	}
	
	/** Tests getting the expected actions when no action pairs have
	 * been logged.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 * 
	 */
	@Test
	public void getAllExpectedActions_empty(){
		assertEquals(this.dummysw.getAllExpectedActions().size(), 0);
	}
	
	/** Tests getting the expected actions when all actions are
	 * both in the expected (row) and actual (col)
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getAllExpectedActions_expectedAndActual() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "thirdAction", "oneAction");
		addPair.invoke(this.dummysw, "thirdAction", "otherAction");
		
		List<String> allExpected = this.dummysw.getAllExpectedActions();
		assertEquals(allExpected.size(), 3);
		assertTrue(allExpected.contains("oneAction"));
		assertTrue(allExpected.contains("otherAction"));
		assertTrue(allExpected.contains("thirdAction"));
	}
	
	/** Tests getting the expected actions when not all actions are
	 * both in the expected (row) and actual (col)
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getAllExpectedActions_expectedAndActualNotSame() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "fourthAction");
		addPair.invoke(this.dummysw, "otherAction", "lastAction");
		addPair.invoke(this.dummysw, "thirdAction", "oneAction");
		addPair.invoke(this.dummysw, "thirdAction", "otherAction");
		
		List<String> allExpected = this.dummysw.getAllExpectedActions();
		assertEquals(allExpected.size(), 3);
		assertTrue(allExpected.contains("oneAction"));
		assertTrue(allExpected.contains("otherAction"));
		assertTrue(allExpected.contains("thirdAction"));
	}
	
	/** Tests getting the expected actions and then adding more after
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getAllExpectedActions_addMore() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "fourthAction");
		addPair.invoke(this.dummysw, "otherAction", "lastAction");
		addPair.invoke(this.dummysw, "thirdAction", "oneAction");
		addPair.invoke(this.dummysw, "thirdAction", "otherAction");
		
		List<String> allExpected = this.dummysw.getAllExpectedActions();
		assertEquals(allExpected.size(), 3);
		assertTrue(allExpected.contains("oneAction"));
		assertTrue(allExpected.contains("otherAction"));
		assertTrue(allExpected.contains("thirdAction"));
		
		addPair.invoke(this.dummysw, "fourthAction", "otherAction");
		addPair.invoke(this.dummysw, "someotherAction", "someotherAction");
		allExpected = this.dummysw.getAllExpectedActions();
		assertEquals(allExpected.size(), 5);
		assertTrue(allExpected.contains("oneAction"));
		assertTrue(allExpected.contains("otherAction"));
		assertTrue(allExpected.contains("thirdAction"));
		assertTrue(allExpected.contains("fourthAction"));
		assertTrue(allExpected.contains("someotherAction"));
	}
		
	/** Tests the method with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getRecall_null(){
		this.dummysw.getRecall(null);
	}
	
	/** Tests the method when the action has never
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getRecall_noAdded(){
		assertEquals(this.dummysw.getRecall("someaction"),0.0f);
	}
	
	/** Tests the method when the action has never
	 * been the expected action
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getRecall_notExpected() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "fourthAction");
		addPair.invoke(this.dummysw, "otherAction", "lastAction");
		
		assertEquals(this.dummysw.getRecall("differentAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered and only it was encountered.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getRecall_validOnlyOne() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getRecall("oneAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "secondAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getRecall("oneAction"), 0.75f);
		assertEquals(this.dummysw.getRecall("secondAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered (along with others).
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getRecall_validMultiple() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		
		assertEquals(this.dummysw.getRecall("oneAction"), 1.0f);
		assertEquals(this.dummysw.getRecall("otherAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "secondAction");
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "oneAction");
		
		assertEquals(this.dummysw.getRecall("oneAction"), 0.5f);
		assertEquals(this.dummysw.getRecall("otherAction"), 0.75f);
		assertEquals(this.dummysw.getRecall("secondAction"), 0.0f);
	}
	
	/** Tests the method with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getPrecision_null(){
		this.dummysw.getPrecision(null);
	}
	
	/** Tests the method when the action has never
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getPrecision_noAdded(){
		assertEquals(this.dummysw.getPrecision("someaction"),0.0f);
	}
	
	/** Tests the method when the action has never
	 * been the expected action
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getPrecision_notExpected() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "fourthAction");
		addPair.invoke(this.dummysw, "otherAction", "lastAction");
		
		assertEquals(this.dummysw.getPrecision("differentAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered and only it was encountered.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getPrecision_validOnlyOne() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getPrecision("oneAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "secondAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getPrecision("oneAction"), 1.0f);
		assertEquals(this.dummysw.getPrecision("secondAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered (along with others).
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getPrecision_validMultiple() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		
		assertEquals(this.dummysw.getPrecision("oneAction"), 1.0f);
		assertEquals(this.dummysw.getPrecision("otherAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "oneAction");
		addPair.invoke(this.dummysw, "xxxAction", "secondAction");
		
		
		assertEquals(this.dummysw.getPrecision("oneAction"), 0.75f);
		assertEquals(this.dummysw.getPrecision("otherAction"), 0.75f);
		assertEquals(this.dummysw.getPrecision("secondAction"), 0.0f);
	}
	

	/** Tests the method with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getF1_null(){
		this.dummysw.getF1(null);
	}
	
	/** Tests the method when the action has never
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getF1_noAdded(){
		assertEquals(this.dummysw.getF1("someaction"),0.0f);
	}
	
	/** Tests the method when the action has never
	 * been the expected action
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getF1_notExpected() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "fourthAction");
		addPair.invoke(this.dummysw, "otherAction", "lastAction");
		
		assertEquals(this.dummysw.getF1("differentAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered and only it was encountered.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getF1_validOnlyOne() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getF1("oneAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "secondAction");
		addPair.invoke(this.dummysw, "oneAction", "secondAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		
		assertEquals(this.dummysw.getF1("oneAction"), 0.75f);
		assertEquals(this.dummysw.getF1("secondAction"), 0.0f);
	}
	
	/** Tests the method when the action has 
	 * been encountered (along with others).
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getF1_validMultiple() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		
		assertEquals(this.dummysw.getF1("oneAction"), 1.0f);
		assertEquals(this.dummysw.getF1("otherAction"), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "oneAction");
		addPair.invoke(this.dummysw, "xxxAction", "secondAction");
		
		
		assertEquals(this.dummysw.getF1("oneAction"), 0.75f);
		assertEquals(this.dummysw.getF1("otherAction"), 0.75f);
		assertEquals(this.dummysw.getF1("secondAction"), 0.0f);
		assertEquals(this.dummysw.getF1("xxxAction"), 0.0f);
	}
	
	/** Tests the method when the action has never
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getGlobalF1_noAdded(){
		assertEquals(this.dummysw.getGlobalF1(),0.0f);
	}
	
	
	/** Tests the method when the action has 
	 * been encountered (along with others).
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.3
	 */
	@Test
	public void getGlobalF1_validMultiple() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		
		assertEquals(this.dummysw.getGlobalF1(), 1.0f);
		
		addPair.invoke(this.dummysw, "oneAction", "otherAction");
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "oneAction");
		addPair.invoke(this.dummysw, "xxxAction", "secondAction");
		
		
		assertEquals(this.dummysw.getGlobalF1(), 0.375f);
	}
	
	/** Tests the method when no actions have
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.5
	 */
	@Test
	public void getStatisticsBundle_noStats(){
		StatisticsBundle sb = this.dummysw.getStatisticsBundle();
		String[] labels = sb.getLabels();
		float[] stats = sb.getAllStatistics();
		assertEquals(labels.length, stats.length);
		assertEquals(labels.length, 2);
		assertEquals(labels[0], "Global F1");
		assertEquals(labels[1], "Classification Accuracy");
		assertEquals(stats[0], 0.0f);
		assertEquals(stats[1], 0.0f);
	}
	
	/** Tests the method when some actions have
	 * been encountered.
	 * 
	 * @author Michael W. Floyd
	 * @throws Exception 
	 * @since 0.5
	 */
	@Test
	public void getStatisticsBundle_someStats() throws Exception{
		Method addPair = addPairHack();
		
		//simulate the pairings
		addPair.invoke(this.dummysw, "oneAction", "oneAction");
		addPair.invoke(this.dummysw, "otherAction", "otherAction");
		addPair.invoke(this.dummysw, "xxxAction", "otherAction");
		addPair.invoke(this.dummysw, "otherAction", "oneAction");
		
		StatisticsBundle sb = this.dummysw.getStatisticsBundle();
		
		String[] labels = sb.getLabels();
		float[] stats = sb.getAllStatistics();
		assertEquals(labels.length, stats.length);
		assertEquals(labels.length, 11);
		
		assertEquals(labels[0], "Global F1");
		assertEquals(labels[1], "Classification Accuracy");
		assertEquals(labels[2], "F1 otherAction");
		assertEquals(labels[3], "Precision otherAction");
		assertEquals(labels[4], "Recall otherAction");
		assertEquals(labels[5], "F1 oneAction");
		assertEquals(labels[6], "Precision oneAction");
		assertEquals(labels[7], "Recall oneAction");
		assertEquals(labels[8], "F1 xxxAction");
		assertEquals(labels[9], "Precision xxxAction");
		assertEquals(labels[10], "Recall xxxAction");
		
		assertEquals(stats[0], (3.5/9));	
		assertEquals(stats[1], 0.5f);
		assertEquals(stats[2], 0.5f);
		assertEquals(stats[3], 0.5f);
		assertEquals(stats[4], 0.5f);
		assertEquals(stats[5], (2/3));
		assertEquals(stats[6], 0.5f);
		assertEquals(stats[7], 1.0f);
		assertEquals(stats[8], 0.0f);
		assertEquals(stats[9], 0.0f);
		assertEquals(stats[10], 0.0f);
	}
	
	/** This method is just a hack so we can directly call our private
	 * method and avoid the high overhead of actually simulating these
	 * events. While this is fairly "dirty", it was done because testing
	 * time was limited, and a quick and dirty approach covered more code
	 * than the clean approach.
	 * 
	 * @return the addPair method that can be directly called.
	 * @throws Exception
	 */
	private Method addPairHack() throws Exception{
		
		Class<StatisticsWrapper> c = StatisticsWrapper.class;
		Method addPair = c.getDeclaredMethod("addPair", new Class[]{String.class, String.class});
		addPair.setAccessible(true);
		
		return addPair;
	}
}
