/** FeatureDistancePresortTest.java in the package tests.junit.org.JIFSA.preprocessing.filters.casebasefilter of the JIFSA project.
    Originally created 17-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.preprocessing.filters.casebasefilter;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.SensoryItem;
import org.JIFSA.preprocessing.filters.casebasefilter.FeatureDistancePresort;
import org.JIFSA.sensoryItems.Spatial2DObject;
import org.junit.Test;



/** Unit tests for the org.JIFSA.preprocessing.filters.casebasefilter.FeatureDistancePresort class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class FeatureDistancePresortTest {

	/** Tests giving a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void filter_null(){
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		fdp.filter(null);
	}
	
	/** Tests giving an empty CaseBase.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_empty(){
		//TODO
	}
	
	/** Tests when the Cases are empty.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_emptyCases(){
		//TODO
	}
	
	
	/** Tests the filter when there is only one type
	 * of Spatial2DFeature and they are out of order
	 * (in a single Case).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneCaseOneType(){
		CaseBase cb = new CaseBase();
		
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type1");
		Spatial2DObject obj4 = new Spatial2DObject("type1");
		
		obj1.setDistance(9.4f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(0.1f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),1);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),1);
		
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj2);
		assertEquals(feats.get(1),obj4);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj1);
	}
	
	/** Tests the filter when there are multiple types
	 * of Spatial2DFeature and they are out of order 
	 * (in a single Case).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneCaseMultipleTypes(){
		CaseBase cb = new CaseBase();
		
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj1.setDistance(10.0f);
		obj2.setDistance(0.1f);
		obj3.setDistance(10.2f);
		obj4.setDistance(9.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),1);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),1);
		
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj2);
		assertEquals(feats1.get(1),obj1);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
	}
	
	/** Tests the filter when the features are not Spatial2DObjects
	 * in a single Case
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneCaseOneTypeNotSpatial(){
		CaseBase cb = new CaseBase();
		
		SensoryItem obj1 = new SensoryItem("type1");
		SensoryItem obj2 = new SensoryItem("type1");
		SensoryItem obj3 = new SensoryItem("type1");
		SensoryItem obj4 = new SensoryItem("type1");
		
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),1);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),1);
		
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj1);
		assertEquals(feats.get(1),obj2);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj4);
	}
	
	/** Tests the filter when some of the features are not Spatial2DObjects
	 * and some are in a single Case
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneCaseMultipleTypesSomeSpatialSomeNot(){
		CaseBase cb = new CaseBase();
		
		SensoryItem obj1 = new SensoryItem ("type1");
		SensoryItem  obj2 = new SensoryItem ("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj3.setDistance(10.2f);
		obj4.setDistance(9.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),1);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),1);
		
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj1);
		assertEquals(feats1.get(1),obj2);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
	}

	/** Tests the filter when there is only one type
	 * of Spatial2DFeature and they are out of order
	 * (in multiple Cases).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleCasesOneType(){
		CaseBase cb = new CaseBase();
		
		//case 1
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type1");
		Spatial2DObject obj4 = new Spatial2DObject("type1");
		
		obj1.setDistance(9.4f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(0.1f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		//case 2
		Spatial2DObject obj5 = new Spatial2DObject("type1");
		Spatial2DObject obj6 = new Spatial2DObject("type1");
		Spatial2DObject obj7 = new Spatial2DObject("type1");
		Spatial2DObject obj8 = new Spatial2DObject("type1");
		
		obj5.setDistance(9.4f);
		obj6.setDistance(10.1f);
		obj7.setDistance(0.1f);
		obj8.setDistance(12.4f);
		
		AgentInputs ai2 = new AgentInputs();
		ai2.addSensoryItem(obj5);
		ai2.addSensoryItem(obj6);
		ai2.addSensoryItem(obj7);
		ai2.addSensoryItem(obj8);
		
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		
		Case c2 = new Case(ai2,act2);
		
		cb.addCase(c2);
		
		//sort
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),2);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),2);
		
		//check case 1
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj2);
		assertEquals(feats.get(1),obj4);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj1);
		
		//check case 2
		Case filteredCase2 = cases.get(1);
		
		AgentInputs filteredInputs2 = filteredCase2.getInputs();
		
		assertNotNull(filteredInputs2);
		assertEquals(filteredInputs2.getNumberSensoryItems(),4);
		assertEquals(filteredInputs2.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs2.doesContain("type1"));
		assertEquals(filteredInputs2.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats2 = filteredInputs2.getSensoryItems("type1");
		assertEquals(feats2.get(0),obj7);
		assertEquals(feats2.get(1),obj5);
		assertEquals(feats2.get(2),obj6);
		assertEquals(feats2.get(3),obj8);
	}
	
	/** Tests the filter when there is multiple types
	 * of Spatial2DFeature and they are out of order
	 * (in multiple Cases).
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleCasesMultipleTypes(){
		CaseBase cb = new CaseBase();
		
		//case 1
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj1.setDistance(9.4f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(0.1f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		//case 2
		Spatial2DObject obj5 = new Spatial2DObject("type1");
		Spatial2DObject obj6 = new Spatial2DObject("type2");
		Spatial2DObject obj7 = new Spatial2DObject("type2");
		Spatial2DObject obj8 = new Spatial2DObject("type1");
		
		obj5.setDistance(9.4f);
		obj6.setDistance(10.1f);
		obj7.setDistance(0.1f);
		obj8.setDistance(12.4f);
		
		AgentInputs ai2 = new AgentInputs();
		ai2.addSensoryItem(obj5);
		ai2.addSensoryItem(obj6);
		ai2.addSensoryItem(obj7);
		ai2.addSensoryItem(obj8);
		
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		
		Case c2 = new Case(ai2,act2);
		
		cb.addCase(c2);
		
		//sort
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),2);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),2);
		
		//check case 1
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj2);
		assertEquals(feats1.get(1),obj1);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
		
		//check case 2
		Case filteredCase2 = cases.get(1);
		
		AgentInputs filteredInputs2 = filteredCase2.getInputs();
		
		assertNotNull(filteredInputs2);
		assertEquals(filteredInputs2.getNumberSensoryItems(),4);
		assertEquals(filteredInputs2.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs2.doesContain("type1"));
		assertTrue(filteredInputs2.doesContain("type2"));
		assertEquals(filteredInputs2.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs2.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats3 = filteredInputs2.getSensoryItems("type1");
		assertEquals(feats3.get(0),obj5);
		assertEquals(feats3.get(1),obj8);
		List<SensoryItem> feats4 = filteredInputs2.getSensoryItems("type2");
		assertEquals(feats4.get(0),obj7);
		assertEquals(feats4.get(1),obj6);
	}
	
	/** Tests the filter when the features are not Spatial2DObjects
	 * in multiple Cases
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_MultipleCasesOneTypeNotSpatial(){
		CaseBase cb = new CaseBase();
		
		//case 1
		SensoryItem obj1 = new SensoryItem("type1");
		SensoryItem obj2 = new SensoryItem("type1");
		SensoryItem obj3 = new SensoryItem("type1");
		SensoryItem obj4 = new SensoryItem("type1");
		
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		//case 2
		SensoryItem obj5 = new SensoryItem("type1");
		SensoryItem obj6 = new SensoryItem("type1");
		SensoryItem obj7 = new SensoryItem("type1");
		SensoryItem obj8 = new SensoryItem("type1");
		
		
		AgentInputs ai2 = new AgentInputs();
		ai2.addSensoryItem(obj5);
		ai2.addSensoryItem(obj6);
		ai2.addSensoryItem(obj7);
		ai2.addSensoryItem(obj8);
		
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		
		Case c2 = new Case(ai2,act2);
		
		cb.addCase(c2);
		
		//sort
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),2);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),2);
		
		//check case 1
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj1);
		assertEquals(feats.get(1),obj2);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj4);
		
		//check case 2
		Case filteredCase2 = cases.get(1);
		
		AgentInputs filteredInputs2 = filteredCase2.getInputs();
		
		assertNotNull(filteredInputs2);
		assertEquals(filteredInputs2.getNumberSensoryItems(),4);
		assertEquals(filteredInputs2.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs2.doesContain("type1"));
		assertEquals(filteredInputs2.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats2 = filteredInputs2.getSensoryItems("type1");
		assertEquals(feats2.get(0),obj5);
		assertEquals(feats2.get(1),obj6);
		assertEquals(feats2.get(2),obj7);
		assertEquals(feats2.get(3),obj8);
	}


	/** Tests the filter when some of the features are not Spatial2DObjects
	 * and some are in multiple Cases
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleCasesMultipleTypesSomeSpatialSomeNot(){
		CaseBase cb = new CaseBase();
		
		//first case
		SensoryItem obj1 = new SensoryItem ("type1");
		SensoryItem  obj2 = new SensoryItem ("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj3.setDistance(10.2f);
		obj4.setDistance(9.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		cb.addCase(c);
		
		//second case
		SensoryItem obj5 = new SensoryItem ("type3");
		SensoryItem  obj6 = new SensoryItem ("type3");
		Spatial2DObject obj7 = new Spatial2DObject("type2");
		Spatial2DObject obj8 = new Spatial2DObject("type2");
		
		obj7.setDistance(110.2f);
		obj8.setDistance(19.4f);
		
		AgentInputs ai2 = new AgentInputs();
		ai2.addSensoryItem(obj5);
		ai2.addSensoryItem(obj6);
		ai2.addSensoryItem(obj7);
		ai2.addSensoryItem(obj8);
		
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		
		Case c2 = new Case(ai2,act2);
		
		cb.addCase(c2);
		
		//sort
		FeatureDistancePresort fdp = new FeatureDistancePresort();
		
		CaseBase filtered = fdp.filter(cb);
		
		assertEquals(filtered.getCasebaseSize(),2);
		List<Case> cases = filtered.getCaseList();
		assertEquals(cases.size(),2);
		
		//check Case 1
		Case filteredCase = cases.get(0);
		
		AgentInputs filteredInputs = filteredCase.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj1);
		assertEquals(feats1.get(1),obj2);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
		
		//check Case 2
		Case filteredCase2 = cases.get(1);
		
		AgentInputs filteredInputs2 = filteredCase2.getInputs();
		
		assertNotNull(filteredInputs2);
		assertEquals(filteredInputs2.getNumberSensoryItems(),4);
		assertEquals(filteredInputs2.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs2.doesContain("type3"));
		assertTrue(filteredInputs2.doesContain("type2"));
		assertEquals(filteredInputs2.getNumberSensoryItems("type3"),2);
		assertEquals(filteredInputs2.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats3 = filteredInputs2.getSensoryItems("type3");
		assertEquals(feats3.get(0),obj5);
		assertEquals(feats3.get(1),obj6);
		List<SensoryItem> feats4 = filteredInputs2.getSensoryItems("type2");
		assertEquals(feats4.get(0),obj8);
		assertEquals(feats4.get(1),obj7);
	}
}
