/** NoActionsFilterTest.java in the package tests.junit.org.JIFSA.preprocessing.filters.casebasefilter of the JIFSA project.
    Originally created 17-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.preprocessing.filters.casebasefilter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.preprocessing.filters.casebasefilter.NoActionsFilter;
import org.junit.Test;


/** Unit tests for the org.JIFSA.preprocessing.filters.casebasefilter.NoActionsFilter class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class NoActionsFilterTest {

	/** Tests when a null parameter is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void filter_null(){
		NoActionsFilter naf = new NoActionsFilter();
		naf.filter(null);
	}
	
	/** Test supplying the method with an empty case base
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_emptyCaseBase(){
		CaseBase cb = new CaseBase();
		NoActionsFilter naf = new NoActionsFilter();
		
		CaseBase filtered = naf.filter(cb);
		assertNotNull(filtered);
		assertEquals(filtered.getCasebaseSize(),0);
	}
	
	/** Test supplying the method with a case base where all Cases
	 * have no associated actions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_allNoActions(){
		CaseBase cb = new CaseBase();
		Case c1 = new Case();
		Case c2 = new Case();
		Case c3 = new Case();
		cb.addCase(c1);
		cb.addCase(c2);
		cb.addCase(c3);
		
		NoActionsFilter naf = new NoActionsFilter();
		CaseBase filtered = naf.filter(cb);
		assertNotNull(filtered);
		assertEquals(filtered.getCasebaseSize(),0);
	}
	
	/** Test supplying the method with a case base where some Cases
	 * have no associated actions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_someNoActions(){
		CaseBase cb = new CaseBase();
		Case c1 = new Case();
		AgentInputs ai2 = new AgentInputs();
		List<AgentAction> ac2 = new ArrayList<AgentAction>();
		ac2.add(new AgentAction("someAction"));
		Case c2 = new Case(ai2,ac2);
		AgentInputs ai3 = new AgentInputs();
		List<AgentAction> ac3 = new ArrayList<AgentAction>();
		ac3.add(new AgentAction("anotherAction"));
		Case c3 = new Case(ai3,ac3);
		Case c4 = new Case();
		cb.addCase(c1);
		cb.addCase(c2);
		cb.addCase(c3);
		cb.addCase(c4);
		
		NoActionsFilter naf = new NoActionsFilter();
		CaseBase filtered = naf.filter(cb);
		assertNotNull(filtered);
		assertEquals(filtered.getCasebaseSize(),2);
		List<Case> filteredCases = filtered.getCaseList();
		assertFalse(filteredCases.contains(c1));
		assertTrue(filteredCases.contains(c2));
		assertTrue(filteredCases.contains(c3));
		assertFalse(filteredCases.contains(c4));
	}
	
	/** Test supplying the method with a case base where no Cases
	 * have no associated actions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_noneNoActions(){
		CaseBase cb = new CaseBase();
		AgentInputs ai1 = new AgentInputs();
		List<AgentAction> ac1 = new ArrayList<AgentAction>();
		ac1.add(new AgentAction("firstAction"));
		Case c1 = new Case(ai1,ac1);
		AgentInputs ai2 = new AgentInputs();
		List<AgentAction> ac2 = new ArrayList<AgentAction>();
		ac2.add(new AgentAction("someAction"));
		Case c2 = new Case(ai2,ac2);
		AgentInputs ai3 = new AgentInputs();
		List<AgentAction> ac3 = new ArrayList<AgentAction>();
		ac3.add(new AgentAction("anotherAction"));
		Case c3 = new Case(ai3,ac3);
		AgentInputs ai4 = new AgentInputs();
		List<AgentAction> ac4 = new ArrayList<AgentAction>();
		ac4.add(new AgentAction("lastAction"));
		Case c4 = new Case(ai4,ac4);
		cb.addCase(c1);
		cb.addCase(c2);
		cb.addCase(c3);
		cb.addCase(c4);
		
		NoActionsFilter naf = new NoActionsFilter();
		CaseBase filtered = naf.filter(cb);
		assertNotNull(filtered);
		assertEquals(filtered.getCasebaseSize(),4);
		List<Case> filteredCases = filtered.getCaseList();
		assertTrue(filteredCases.contains(c1));
		assertTrue(filteredCases.contains(c2));
		assertTrue(filteredCases.contains(c3));
		assertTrue(filteredCases.contains(c4));
	}
	
}
