/** FeatureDistanceCasePresortTest.java in the package tests.junit.org.JIFSA.preprocessing.filters.casefilter of the JIFSA project.
    Originally created 17-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.preprocessing.filters.casefilter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.JIFSA.preprocessing.filters.casefilter.FeatureDistanceCasePresort;
import org.JIFSA.sensoryItems.Spatial2DObject;
import org.junit.Test;



/** Unit tests for the org.JIFSA.preprocessing.filters.casefilter.FeatureDistanceCasePresort class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class FeatureDistanceCasePresortTest {

	/** Tests giving a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void filter_null(){
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		fdcp.filter(null);
	}
	
	/** Tests giving an empty Case.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_empty(){
		Case c = new Case();
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		Case filtered = fdcp.filter(c);
		
		assertNotNull(filtered);
		assertEquals(filtered.getInputs().getNumberSensoryItems(), 0);
		assertEquals(filtered,c);
	}
	/** Tests the filter when there is only one type
	 * of Spatial2DFeature and they are already in order.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneTypeInOrder(){
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type1");
		Spatial2DObject obj4 = new Spatial2DObject("type1");
		
		obj1.setDistance(0.0f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(99.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj1);
		assertEquals(feats.get(1),obj2);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj4);		
	}
	
	/** Tests the filter when there is only one type
	 * of Spatial2DFeature and they are out of order.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneTypeOutOfOrder(){
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type1");
		Spatial2DObject obj4 = new Spatial2DObject("type1");
		
		obj1.setDistance(9.4f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(0.1f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj2);
		assertEquals(feats.get(1),obj4);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj1);
	}
	
	/** Tests the filter when there are multiple types
	 * of Spatial2DFeature and they are in order.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleTypesInOrder(){
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj1.setDistance(0.0f);
		obj2.setDistance(0.1f);
		obj3.setDistance(0.2f);
		obj4.setDistance(99.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj1);
		assertEquals(feats1.get(1),obj2);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj3);
		assertEquals(feats2.get(1),obj4);
	}
	
	/** Tests the filter when there are multiple types
	 * of Spatial2DFeature and they are out of order.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleTypesOutOfOrder(){
		Spatial2DObject obj1 = new Spatial2DObject("type1");
		Spatial2DObject obj2 = new Spatial2DObject("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj1.setDistance(10.0f);
		obj2.setDistance(0.1f);
		obj3.setDistance(10.2f);
		obj4.setDistance(9.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj2);
		assertEquals(feats1.get(1),obj1);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
	}
	
	/** Tests the filter when the features are not Spatial2DObjects
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_oneTypeNotSpatial(){
		SensoryItem obj1 = new SensoryItem("type1");
		SensoryItem obj2 = new SensoryItem("type1");
		SensoryItem obj3 = new SensoryItem("type1");
		SensoryItem obj4 = new SensoryItem("type1");
		
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),1);
		assertTrue(filteredInputs.doesContain("type1"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),4);
		
		List<SensoryItem> feats = filteredInputs.getSensoryItems("type1");
		assertEquals(feats.get(0),obj1);
		assertEquals(feats.get(1),obj2);
		assertEquals(feats.get(2),obj3);
		assertEquals(feats.get(3),obj4);
	}
	
	/** Tests the filter when some of the features are not Spatial2DObjects
	 * and some are.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void filter_multipleTypesSomeSpatialSomeNot(){
		SensoryItem obj1 = new SensoryItem ("type1");
		SensoryItem  obj2 = new SensoryItem ("type1");
		Spatial2DObject obj3 = new Spatial2DObject("type2");
		Spatial2DObject obj4 = new Spatial2DObject("type2");
		
		obj3.setDistance(10.2f);
		obj4.setDistance(9.4f);
		
		AgentInputs ai = new AgentInputs();
		ai.addSensoryItem(obj1);
		ai.addSensoryItem(obj2);
		ai.addSensoryItem(obj3);
		ai.addSensoryItem(obj4);
		
		List<AgentAction> act = new ArrayList<AgentAction>();
		
		Case c = new Case(ai,act);
		
		FeatureDistanceCasePresort fdcp = new FeatureDistanceCasePresort();
		
		Case filtered = fdcp.filter(c);
		
		AgentInputs filteredInputs = filtered.getInputs();
		
		assertNotNull(filteredInputs);
		assertEquals(filteredInputs.getNumberSensoryItems(),4);
		assertEquals(filteredInputs.getSensoryItemNames().size(),2);
		assertTrue(filteredInputs.doesContain("type1"));
		assertTrue(filteredInputs.doesContain("type2"));
		assertEquals(filteredInputs.getNumberSensoryItems("type1"),2);
		assertEquals(filteredInputs.getNumberSensoryItems("type2"),2);
		
		List<SensoryItem> feats1 = filteredInputs.getSensoryItems("type1");
		assertEquals(feats1.get(0),obj1);
		assertEquals(feats1.get(1),obj2);
		List<SensoryItem> feats2 = filteredInputs.getSensoryItems("type2");
		assertEquals(feats2.get(0),obj4);
		assertEquals(feats2.get(1),obj3);
	}
	
	
}
