/** ParsedMessageTest.java in the package tests.junit.org.JIFSA.preprocessing.parsers of the JIFSA project.
    Originally created 30-Sep-08

    Copyright (C) 2008 Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.preprocessing.parsers;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.Set;

import org.JIFSA.preprocessing.parsers.ParsedMessage;
import org.junit.Test;

/** Tests for the org.JIFSA.preprocessing.parsers.ParsedMessage class.
 * 
 * @author Michael W. Floyd
 * @since 0.4
 *
 */
public class ParsedMessageTest {

	/** Tests the constructor to see that default values
	 * are properly set
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void constructor_defaults(){
		ParsedMessage pm = new ParsedMessage();
		assertEquals(pm.getType(), ParsedMessage.NONE);
		assertEquals(pm.getSubtype(), ParsedMessage.NONE);
		assertEquals(pm.nParams(), 0);
		assertFalse(pm.hasParams());
	}
	
	/** Tests the reset function when nothing has been done
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void reset_noneadded(){
		ParsedMessage pm = new ParsedMessage();
		assertEquals(pm.getType(), ParsedMessage.NONE);
		assertEquals(pm.getSubtype(), ParsedMessage.NONE);
		assertEquals(pm.nParams(), 0);
		assertFalse(pm.hasParams());
		pm.reset();
		assertEquals(pm.getType(), ParsedMessage.NONE);
		assertEquals(pm.getSubtype(), ParsedMessage.NONE);
		assertEquals(pm.nParams(), 0);
		assertFalse(pm.hasParams());
	}
	
	/** Tests the reset function after it has been changed
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void reset_someadded(){
		ParsedMessage pm = new ParsedMessage();
		pm.setType("Sometype");
		pm.setSubtype("asubtype");
		pm.setParam("randomParam", "aValue");
		
		assertEquals(pm.getType(), "Sometype");
		assertEquals(pm.getSubtype(), "asubtype");
		assertEquals(pm.nParams(), 1);
		assertTrue(pm.hasParams());
		pm.reset();
		assertEquals(pm.getType(), ParsedMessage.NONE);
		assertEquals(pm.getSubtype(), ParsedMessage.NONE);
		assertEquals(pm.nParams(), 0);
		assertFalse(pm.hasParams());
	}
	
	/** Tests setting a valid type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void setType_valid(){
		ParsedMessage pm = new ParsedMessage();
		assertFalse(pm.getType().equals("sometype"));
		pm.setType("sometype");
		assertEquals(pm.getType(), "sometype");
	}
	
	/** Tests setting a null type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setType_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.setType(null);
	}
	
	/** Tests setting a null sensor message
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setSensor_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.setSensor(null);
	}
	
	/** Tests setting a valid sensor message
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void setSensor_valid(){
		ParsedMessage pm = new ParsedMessage();
		pm.setSensor("somemessage");
		assertEquals(pm.getSubtype(), "somemessage");
		assertEquals(pm.getType(), ParsedMessage.TYPE_SENSOR);
		assertEquals(pm.getSubtype(), "somemessage");
	}
	
	/** Tests setting a null control message
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setControl_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.setControl(null);
	}
	
	/** Tests setting a valid control message
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void setControl_valid(){
		ParsedMessage pm = new ParsedMessage();
		pm.setControl("amessage");
		assertEquals(pm.getSubtype(), "amessage");
		assertEquals(pm.getType(), ParsedMessage.TYPE_CONTROL);
		assertEquals(pm.getSubtype(), "amessage");
	}
	
	/** Tests setting a null message subtype
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setSubtype_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.setSubtype(null);
	}
	
	/** Tests setting a valid message subtype
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void setSubtype_valid(){
		ParsedMessage pm = new ParsedMessage();
		pm.setSubtype("anewmessage");
		assertEquals(pm.getSubtype(), "anewmessage");
		assertEquals(pm.getType(), ParsedMessage.NONE);
	}
	
	/** Tests setting a a null parameter type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setParameters_nullParam(){
		ParsedMessage pm = new ParsedMessage();
		pm.setParam(null, "valuevalue");
	}
	
	/** Tests setting a a null parameter value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setParameters_nullValue(){
		ParsedMessage pm = new ParsedMessage();
		pm.setParam("someparamparam", null);
	}
	
	/** Tests getting a null parameter type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getValues_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.getValue(null);
	}
	
	/** Tests setting a a valid parameter type/value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test
	public void setParameters_valid(){
		ParsedMessage pm = new ParsedMessage();
		pm.setParam("param1", "aValue");
		pm.setParam("param2", "anotherValue");
		pm.setParam("param3", "lastValue");
		
		assertEquals(pm.nParams(), 3);
		assertTrue(pm.contains("param1"));
		assertEquals(pm.getValue("param1"), "aValue");
		assertTrue(pm.contains("param2"));
		assertEquals(pm.getValue("param2"), "anotherValue");
		assertTrue(pm.contains("param3"));
		assertEquals(pm.getValue("param3"), "lastValue");
		
		Set<String> params = pm.params();
		assertEquals(params.size(), 3);
		assertTrue(params.contains("param1"));
		assertTrue(params.contains("param2"));
		assertTrue(params.contains("param3"));
	}
	
	/** Tests using a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.4
	 */
	@Test(expected=IllegalArgumentException.class)
	public void contains_null(){
		ParsedMessage pm = new ParsedMessage();
		pm.contains(null);
	}
	
}
