/** WeightsTest.java in the package tests.junit.org.JIFSA.reasoning of the JIFSA project.
    Originally created 21-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.reasoning;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.JIFSA.reasoning.Weights;
import org.junit.Test;

/** Tests for the org.jCIFSAA.reasoning.Weights class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class WeightsTest {

	/** Tests the constructor with a default
	 * weight that is less than zero.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_lessThanZero(){
		new Weights(-0.0001f);
	}
	
	/** Tests the constructor with a default
	 * weight that is greater than one.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_greaterThanOne(){
		new Weights(1.00001f);
	}
	
	/** Tests the constructor with weights
	 * that are on the boundary of acceptable
	 * values are well are other valid values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void constructor_validDefaults(){
		Weights w0 = new Weights(0.0f);
		Weights w1 = new Weights(1.0f);
		Weights w2 = new Weights(0.37f);
		Weights w3 = new Weights(0.88f);
		
		List<String> types = w0.getWeightedItemNames();
		assertEquals(types.size(),0);
		types = w1.getWeightedItemNames();
		assertEquals(types.size(),0);
		types = w2.getWeightedItemNames();
		assertEquals(types.size(),0);
		types = w3.getWeightedItemNames();
		assertEquals(types.size(),0);
		
		float weight = w0.getWeight("NotInList");
		assertEquals(weight, 0.0f);
		weight = w1.getWeight("NotInList");
		assertEquals(weight, 1.0f);
		weight = w2.getWeight("NotInList");
		assertEquals(weight, 0.37f);
		weight = w3.getWeight("NotInList");
		assertEquals(weight, 0.88f);
	}
	
	/** Tests the method when a null
	 * parameter is given.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setWeight_nullName(){
		Weights w = new Weights(0.0f);
		w.setWeight(null, 1.0f);
	}
	
	/** Tests the method when the supplied
	 * weight is less than 0.0
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setWeight_lessThanZero(){
		Weights w = new Weights(0.0f);
		w.setWeight("myItem", -0.0001f);
	}
	
	/** Tests the method when the supplied
	 * weight is greater than 1.0
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setWeight_greaterThanOne(){
		Weights w = new Weights(0.0f);
		w.setWeight("myItem", 1.0001f);
	}
	
	/** Tests the get/set methods setting the weights
	 * to the boundary values as well as
	 * valid values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void setWeight_getWeight_boundaryAndValid(){
		Weights w = new Weights(0.5f);
		w.setWeight("firstItem", 0.0f);
		w.setWeight("secondItem", 1.0f);
		w.setWeight("thirdItem", 0.44f);
		w.setWeight("fourthItem", 0.87f);
		assertEquals(w.getWeight("firstItem"),0.0f);
		assertEquals(w.getWeight("secondItem"),1.0f);
		assertEquals(w.getWeight("thirdItem"),0.44f);
		assertEquals(w.getWeight("fourthItem"),0.87f);
	}
	
	
	/** Gets the list of weighted items when none have been
	 * set yet.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getWeightedItemNames_default(){
		Weights w = new Weights(0.43f);
		List<String> items = w.getWeightedItemNames();
		assertEquals(items.size(),0);
	}
	
	/** Gets the list of weighted items when one
	 * has been set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getWeightedItemNames_singleType(){
		Weights w = new Weights(0.78f);
		w.setWeight("anItem", 0.22f);
		List<String> items = w.getWeightedItemNames();
		assertEquals(items.size(), 1);
		assertTrue(items.contains("anItem"));
	}
	
	/** Gets the list of weighted items when multiple
	 * types have been set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getWeightedItemNames_multipleTypes(){
		Weights w = new Weights(0.78f);
		w.setWeight("anItem", 0.22f);
		w.setWeight("secondItem", 0.01f);
		w.setWeight("lastItem", 0.88f);
		w.setWeight("secondItem", 0.03f);
		List<String> items = w.getWeightedItemNames();
		assertEquals(items.size(), 3);
		assertTrue(items.contains("anItem"));
		assertTrue(items.contains("secondItem"));
		assertTrue(items.contains("lastItem"));
	}
	
	/** Tests the method with a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void getWeight_nullName(){
		Weights w = new Weights(0.0f);
		w.getWeight(null);
	}
	
	/** Tests the method on an item type
	 * that has not been set yet.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void getWeight_notAdded(){
		Weights w = new Weights(0.476f);
		List<String> items = w.getWeightedItemNames();
		assertFalse(items.contains("someItem"));
		assertEquals(w.getWeight("someItem"),0.476f);
	}
	
	/** Tests the equals method with a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_null(){
		Weights w = new Weights(1.0f);
		assertFalse(w.equals(null));
	}
	
	/** Tests the equals method with itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_self(){
		Weights w = new Weights(1.0f);
		assertTrue(w.equals(w));
	}
	
	/** Tests the equals method with unequal number of weights
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_differentNumberWeights(){
		Weights w1 = new Weights(1.0f);
		Weights w2 = new Weights(1.0f);
		
		w1.setWeight("w1", 1.0f);
		w1.setWeight("w2", 1.0f);
		w1.setWeight("w3", 1.0f);
		
		w2.setWeight("w1", 1.0f);
		w2.setWeight("w2", 1.0f);
		
		assertFalse(w1.equals(w2));
	}
	
	/** Tests the equals method with unequal weights
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_differentWeights(){
		Weights w1 = new Weights(1.0f);
		Weights w2 = new Weights(1.0f);
		
		w1.setWeight("w1", 0.1f);
		w1.setWeight("w2", 1.0f);
		
		w2.setWeight("w1", 1.0f);
		w2.setWeight("w2", 1.0f);
		
		assertFalse(w1.equals(w2));
	}
	
	/** Tests the equals method with same number, but different types
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_sameNumberDifferentType(){
		Weights w1 = new Weights(1.0f);
		Weights w2 = new Weights(1.0f);
		
		w1.setWeight("w1", 1.0f);
		w1.setWeight("w2", 1.0f);
		
		w2.setWeight("w1", 1.0f);
		w2.setWeight("w3", 1.0f);
		
		assertFalse(w1.equals(w2));
	}
	
	/** Tests the equals method with equal weights
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_sameWeights(){
		Weights w1 = new Weights(1.0f);
		Weights w2 = new Weights(1.0f);
		
		w1.setWeight("w1", 0.1f);
		w1.setWeight("w2", 0.6f);
		
		w2.setWeight("w1", 0.1f);
		w2.setWeight("w2", 0.6f);
		
		assertTrue(w1.equals(w2));
	}
	
	/** Tests the equals method with equal weights
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_default(){
		Weights w1 = new Weights(0.7f);
		Weights w2 = new Weights(0.7f);
		
		assertTrue(w1.equals(w2));
	}
	
	/** Tests the equals when they have a different default
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_differntDefault(){
		Weights w1 = new Weights(0.7f);
		Weights w2 = new Weights(0.6f);
		
		w1.setWeight("w1", 0.1f);
		w1.setWeight("w2", 0.6f);
		
		w2.setWeight("w1", 0.1f);
		w2.setWeight("w2", 0.6f);
		
		assertFalse(w1.equals(w2));
	}
	
	/** Tests the equals when they have a different default
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void equals_differntClass(){
		Weights w = new Weights(1.0f);
		
		assertFalse(w.equals("string"));
	}
	
	/** Tests toString with a default weight
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void toString_default(){
		Weights w = new Weights(1.0f);
		String tostring = w.toString();
		
		assertEquals(tostring,"{}");
	}
	
	/** Tests toString with a valid weight
	 * 
	 * @author Michael W. Floyd
	 * @since 0.5
	 */
	@Test
	public void toString_valid(){
		Weights w = new Weights(1.0f);
		w.setWeight("item1", 0.1f);
		w.setWeight("item2", 0.4f);
		w.setWeight("item3", 0.8f);
		String tostring = w.toString();
		
		assertTrue(tostring.contains("item1:0.1"));
		assertTrue(tostring.contains("item2:0.4"));
		assertTrue(tostring.contains("item3:0.8"));
		assertTrue(tostring.startsWith("{"));
		assertTrue(tostring.endsWith("}"));
	}
}
