/** ClosestNeighbourSelectionTest.java in the package tests.junit.org.JIFSA.reasoning.actionselection of the JIFSA project.
    Originally created 23-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.reasoning.actionselection;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.junit.Test;

/** Tests for org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class ClosestNeighbourSelectionTest {

	/** Test giving the constructor a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_null(){
		new ClosestNeighbourSelection(null);
	}
	
	/** Test giving the function a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void selectAction_null(){
		ClosestNeighbourSelection cns = new ClosestNeighbourSelection(new LastActionEstimate());
		cns.selectAction(null);
	}
	
	/** Test giving the function an empty list.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void selectAction_Empty(){
		ClosestNeighbourSelection cns = new ClosestNeighbourSelection(new LastActionEstimate());
		List<Case> neighbours = new ArrayList<Case>();
		cns.selectAction(neighbours);
	}
	
	/** Tests when the closest neighbour does not
	 * have an action associated with it
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void selectAction_NoAction(){
		ClosestNeighbourSelection cns = new ClosestNeighbourSelection(new LastActionEstimate());
		List<Case> neighbours = new ArrayList<Case>();
		neighbours.add(new Case());
		assertNull(cns.selectAction(neighbours));
	}
	
	/** Tests when only one nearest neighbour is
	 * given.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void selectAction_OnlyOne(){
		ClosestNeighbourSelection cns = new ClosestNeighbourSelection(new LastActionEstimate());
		List<Case> neighbours = new ArrayList<Case>();
		
		AgentInputs vis = new AgentInputs();
		List<AgentAction> act = new ArrayList<AgentAction>();
		AgentAction a1 = new AgentAction("myAction");
		act.add(a1);
		Case c = new Case(vis,act);
		neighbours.add(c);
		
		AgentAction toPerform = cns.selectAction(neighbours);
		assertEquals(toPerform, a1);
	}
	
	/** Tests when several nearest neighbours are
	 * given.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void selectAction_Many(){
		ClosestNeighbourSelection cns = new ClosestNeighbourSelection(new LastActionEstimate());
		List<Case> neighbours = new ArrayList<Case>();
		
		AgentInputs vis1 = new AgentInputs();
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		AgentAction a1 = new AgentAction("myAction");
		act1.add(a1);
		Case c1 = new Case(vis1,act1);
		neighbours.add(c1);
		
		AgentInputs vis2 = new AgentInputs();
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		AgentAction a2 = new AgentAction("otherAction");
		act2.add(a2);
		Case c2 = new Case(vis2,act2);
		neighbours.add(c2);
		
		AgentInputs vis3 = new AgentInputs();
		List<AgentAction> act3 = new ArrayList<AgentAction>();
		AgentAction a3 = new AgentAction("thirdAction");
		act3.add(a3);
		Case c3 = new Case(vis3,act3);
		neighbours.add(c3);
		
		AgentAction toPerform = cns.selectAction(neighbours);
		assertEquals(toPerform, a1);
	}
	
}
