/** NearestNeighbourSearchTest.java in the package tests.junit.org.JIFSA.reasoning.casebasesearch of the JIFSA project.
    Originally created Nov 23, 2007

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.reasoning.casebasesearch;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithm;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.penalty.ConstantPenalty;

import org.junit.Before;
import org.junit.Test;

/** Tests for the org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class NearestNeighbourSearchTest {

	private Case c1;
	private Case c2;
	private Case c3;
	private Case c4;
	private Case c5;
	
	/** Creates the distance measure
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void initializeAlgorithms(){
		Weights w = new Weights(1.0f);	
		Case.setGlobalDistanceCalculation(new OrderIndexMatchingAlgorithm(w));
		
		SensoryItem.setPenaltyDistanceCalculation(new ConstantPenalty(1000));
		SensoryItem.setDistanceCalculation(new EqualityDistanceAlgorithm());
	}
	
	/** Creates the Cases that are used for the CaseBase
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Before
	public void initializeCases(){
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("featureOne");
		av1.addSensoryItem(f1);
		List<AgentAction> actions1 = new ArrayList<AgentAction>();
		this.c1 = new Case(av1, actions1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f2 = new SensoryItem("featureTwo");
		av2.addSensoryItem(f2);
		List<AgentAction> actions2 = new ArrayList<AgentAction>();
		this.c2 = new Case(av2, actions2);
		
		AgentInputs av3 = new AgentInputs();
		SensoryItem f3 = new SensoryItem("featureOne");
		av3.addSensoryItem(f3);
		List<AgentAction> actions3 = new ArrayList<AgentAction>();
		this.c3 = new Case(av3, actions3);
	
		AgentInputs av4 = new AgentInputs();
		SensoryItem f41 = new SensoryItem("featureOne");
		SensoryItem f42 = new SensoryItem("featureOne");
		av4.addSensoryItem(f41);
		av4.addSensoryItem(f42);
		List<AgentAction> actions4 = new ArrayList<AgentAction>();
		this.c4 = new Case(av4, actions4);
		
		AgentInputs av5 = new AgentInputs();
		SensoryItem f51 = new SensoryItem("featureOne");
		SensoryItem f52 = new SensoryItem("featureTwo");
		SensoryItem f53 = new SensoryItem("featureOne");
		av5.addSensoryItem(f51);
		av5.addSensoryItem(f52);
		av5.addSensoryItem(f53);
		List<AgentAction> actions5 = new ArrayList<AgentAction>();
		this.c5 = new Case(av5, actions5);
	}
		
	/** Tests the method when the number of
	 * Cases to return is invalid (zero or less)
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void constructor_invalidKZero(){
		new NearestNeighbourSearch(0);
	}
	
	/** Tests the method when the number of
	 * Cases to return is invalid (zero or less)
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void constructor_invalidKNegative(){
		new NearestNeighbourSearch(-1);
	}
	
	/** Tests the method when a parameter is null.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void findClosest_nullBoth(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(1);
		nns.findClosest(null, null);
	}
	
	/** Tests the method when a parameter is null.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void findClosest_nullCase(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(1);
		nns.findClosest(null, new CaseBase());
	}
	
	/** Tests the method when a parameter is null.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class) 
	public void findClosest_nullCaseBase(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(1);
		nns.findClosest(new Case(), null);
	}
	
	/** Tests when the CaseBase contains fewer
	 * Cases then the number of neighbours to find.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_KLargerCaseBaseSize(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(5);
		
		CaseBase cb = new CaseBase();
		cb.addCase(this.c1);
		cb.addCase(this.c2);
		
		List<Case> nearest = nns.findClosest(this.c3, cb);
		assertEquals(nearest.size(),2);
		assertTrue(nearest.contains(this.c1));
		assertTrue(nearest.contains(this.c2));
	}
	
	/** Tests when the CaseBase contains zero
	 * Cases.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_noCases(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(3);
		List<Case> nearest = nns.findClosest(this.c1, new CaseBase());
		assertEquals(nearest.size(),0);
	}
	
	/** Tests when the two Cases are equally
	 * close, but only one nearest neighbour is
	 * required.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_TwoPossibleBest(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(1);
		
		CaseBase cb = new CaseBase();
		cb.addCase(this.c1);
		cb.addCase(this.c2);
		cb.addCase(this.c3);
		cb.addCase(this.c5);
		
		List<Case> nearest = nns.findClosest(this.c4, cb);
		assertEquals(nearest.size(),1);
		assertTrue(nearest.contains(this.c1));
	}
	
	/** Tests 1-NN search
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_KisOne(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(1);
		
		CaseBase cb = new CaseBase();
		cb.addCase(this.c1);
		cb.addCase(this.c2);
		cb.addCase(this.c4);
		cb.addCase(this.c5);
		
		List<Case> nearest = nns.findClosest(this.c3, cb);
		assertEquals(nearest.size(),1);
		assertTrue(nearest.contains(this.c1));
	}
	
	/** Tests k-NN search when k > 1
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_KLargerThanOne(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(3);
		
		CaseBase cb = new CaseBase();
		cb.addCase(this.c1);
		cb.addCase(this.c2);
		cb.addCase(this.c4);
		cb.addCase(this.c5);
		
		List<Case> nearest = nns.findClosest(this.c3, cb);
		assertEquals(nearest.size(),3);
		assertTrue(nearest.contains(this.c1));
		assertTrue(nearest.contains(this.c4));
		assertTrue(nearest.contains(this.c5));
	}
	
	/** Tests k-NN search when k = case base size
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test 
	public void findClosest_KEqualCaseBaseSize(){
		NearestNeighbourSearch nns =  new NearestNeighbourSearch(4);
		
		CaseBase cb = new CaseBase();
		cb.addCase(this.c1);
		cb.addCase(this.c2);
		cb.addCase(this.c4);
		cb.addCase(this.c5);
		
		List<Case> nearest = nns.findClosest(this.c3, cb);
		assertEquals(nearest.size(),4);
		assertTrue(nearest.contains(this.c1));
		assertTrue(nearest.contains(this.c2));
		assertTrue(nearest.contains(this.c4));
		assertTrue(nearest.contains(this.c5));
	}

}
