/** EqualityDistanceAlgorithmTest.java in the package tests.junit.org.JIFSA.reasoning.distance of the JIFSA project.
    Originally created 11-Mar-08

    Copyright (C) 2007 - 2008  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.reasoning.distance;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.junit.Test;

/** Tests for the org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm class.
 * 
 * @author Michael W. Floyd
 * @since 0.3
 *
 */

public class EqualityDistanceAlgorithmTest {
	
	/** Gives a null parameter to the pairwiseDistance
	 * function
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullFirstFeature(){
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		
		SensoryItem f = new SensoryItem("feat");
		
		edm.pairwiseDistance(null, f);
	}
	
	/** Gives a null parameter to the pairwiseDistance
	 * function
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullSecondFeature(){
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		
		SensoryItem f = new SensoryItem("feat");
		
		edm.pairwiseDistance(f, null);
	}
	
	/** Gives a null parameter to the pairwiseDistance
	 * function
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullBothFeatures(){
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		
		edm.pairwiseDistance(null, null);
	}
	
	/** Tests the distance when the features are equal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test
	public void pairwiseDistance_equal(){
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		
		assertEquals(f1,f2);
		assertEquals(edm.pairwiseDistance(f1, f2), EqualityDistanceAlgorithm.MINDISTANCE);
	}
	
	/** Tests the distance when the features are unequal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 *
	 */
	@Test
	public void pairwiseDistance_unequal(){
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("notMyFeature");
		
		assertFalse(f1.equals(f2));
		assertEquals(edm.pairwiseDistance(f1, f2), EqualityDistanceAlgorithm.MAXDISTANCE);
	}

}
