/** OrderIndexMatchingAlgorithmTest.java in the package tests.junit.org.JIFSA.reasoning.distance.globaldistance of the JIFSA project.
    Originally created 16-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.reasoning.distance.globaldistance;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithm;
import org.JIFSA.reasoning.distance.penalty.ConstantPenalty;
import org.junit.Test;

/** Tests for the org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithmTest class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class OrderIndexMatchingAlgorithmTest {

	/** Test giving the constructor a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_null(){
		new OrderIndexMatchingAlgorithm(null);
	}
	
	
	/** Test giving the function a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_firstNull(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		Case c1 = new Case();
		oima.pairwiseDistance(null, c1);
	}
	
	/** Test giving the function a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_secondNull(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		Case c1 = new Case();
		oima.pairwiseDistance(c1, null);
	}
	
	/** Test giving the function a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_bothNull(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		oima.pairwiseDistance(null, null);
	}
	
		
	/** Tests the distance of two empty
	 * Cases.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_emptyCases(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		Case c1 = new Case();
		Case c2 = new Case();
		float distance = oima.pairwiseDistance(c1, c2);
		assertEquals(distance,0.0f);
	}
	
	/** Tests the distance of two non-empty
	 * Cases with an equal number of a single
	 * type of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_equalNumberOneType(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f3 = new SensoryItem("myFeature");
		SensoryItem f4 = new SensoryItem("myFeature");
		av2.addSensoryItem(f3);
		av2.addSensoryItem(f4);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(999);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//they should have all had 0 distance from each other
		assertEquals(distance,0.0f);	
	}
	
	/** Tests the matching of two non-empty
	 * Cases with an unequal number of a single
	 * type of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_unequalNumberOneTypeFirstCaseBigger(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f3 = new SensoryItem("myFeature");
		av2.addSensoryItem(f3);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been one distance 0, one penalty of 1000, for an average of 500
		assertEquals(distance,500f);
	}
	
	/** Tests the matching of two non-empty
	 * Cases with an unequal number of a single
	 * type of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_unequalNumberOneTypeSecondCaseBigger(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f2 = new SensoryItem("myFeature");
		SensoryItem f3 = new SensoryItem("myFeature");
		av2.addSensoryItem(f2);
		av2.addSensoryItem(f3);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been one distance 0, one penalty of 1000, for an average of 500
		assertEquals(distance,500f);
	}

	/** Tests the matching of two non-empty
	 * Cases with an equal number of multiple
	 * types of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_equalNumberMultipleType(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myOtherFeature");
		SensoryItem f3 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		av1.addSensoryItem(f3);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f4 = new SensoryItem("myFeature");
		SensoryItem f5 = new SensoryItem("myFeature");
		SensoryItem f6 = new SensoryItem("myOtherFeature");
		av2.addSensoryItem(f4);
		av2.addSensoryItem(f5);
		av2.addSensoryItem(f6);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been two distances of zero
		assertEquals(distance,0.0f);
	}
	
	
	/** Tests the matching of two non-empty
	 * Cases with an unequal number of one of the
	 * types of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_unequalNumberMultipleType1(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myOtherFeature");
		SensoryItem f3 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		av1.addSensoryItem(f3);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f5 = new SensoryItem("myFeature");
		SensoryItem f6 = new SensoryItem("myOtherFeature");
		av2.addSensoryItem(f5);
		av2.addSensoryItem(f6);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been a distance of 500, plus a distance of 0
		assertEquals(distance,500f);
	}
	
	/** Tests the matching of two non-empty
	 * Cases with an unequal number of one of the
	 * types of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_unequalNumberMultipleType2(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myOtherFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f4 = new SensoryItem("myOtherFeature");
		SensoryItem f5 = new SensoryItem("myFeature");
		SensoryItem f6 = new SensoryItem("myOtherFeature");
		av2.addSensoryItem(f4);
		av2.addSensoryItem(f5);
		av2.addSensoryItem(f6);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been a distance of 500, plus a distance of 0
		assertEquals(distance,500f);
	}
	
	/** Tests the matching of two non-empty
	 * Cases with an unequal number of one of the
	 * types of feature.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_unequalNumberMultipleType3(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myOtherFeature");
		SensoryItem f3 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		av1.addSensoryItem(f3);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f4 = new SensoryItem("myOtherFeature");
		SensoryItem f5 = new SensoryItem("myFeature");
		SensoryItem f6 = new SensoryItem("myOtherFeature");
		av2.addSensoryItem(f4);
		av2.addSensoryItem(f5);
		av2.addSensoryItem(f6);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been a distance of 500, plus a distance of 500
		assertEquals(distance,1000f);
	}
	
	/** Tests the matching of two non-empty
	 * Cases with a different type of feature in one.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_differentType1(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f4 = new SensoryItem("myFeature");
		SensoryItem f5 = new SensoryItem("myFeature");
		SensoryItem f6 = new SensoryItem("myOtherFeature");
		av2.addSensoryItem(f4);
		av2.addSensoryItem(f5);
		av2.addSensoryItem(f6);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been a distance of 0, plus a distance of 1000
		assertEquals(distance,1000f);
	}
	
	/** Tests the matching of two non-empty
	 * Cases with a different type of feature in one.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_differentType2(){
		Weights w = new Weights(1.0f);
		OrderIndexMatchingAlgorithm oima = new OrderIndexMatchingAlgorithm(w);
		
		AgentInputs av1 = new AgentInputs();
		SensoryItem f1 = new SensoryItem("myFeature");
		SensoryItem f2 = new SensoryItem("myFeature");
		SensoryItem f3 = new SensoryItem("myOtherFeature");
		av1.addSensoryItem(f1);
		av1.addSensoryItem(f2);
		av1.addSensoryItem(f3);
		List<AgentAction> act1 = new ArrayList<AgentAction>();
		Case c1 = new Case(av1,act1);
		
		AgentInputs av2 = new AgentInputs();
		SensoryItem f4 = new SensoryItem("myFeature");
		SensoryItem f5 = new SensoryItem("myFeature");
		av2.addSensoryItem(f4);
		av2.addSensoryItem(f5);
		List<AgentAction> act2 = new ArrayList<AgentAction>();
		Case c2 = new Case(av2,act2);
		
		EqualityDistanceAlgorithm edm = new EqualityDistanceAlgorithm();
		ConstantPenalty cp = new ConstantPenalty(1000);
		SensoryItem.setDistanceCalculation(edm);
		SensoryItem.setPenaltyDistanceCalculation(cp);
		float distance = oima.pairwiseDistance(c1, c2);
		
		//there should have been a distance of 0, plus a distance of 1000
		assertEquals(distance,1000f);
	}

}
