/** PolarDistanceAlgorithmTest.java in the package tests.junit.org.JIFSA.reasoning.distance.spatial2D of the JIFSA project.
    Originally created 16-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.reasoning.distance.spatial2D;

import static org.junit.Assert.*;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.spatial2D.PolarDistanceAlgorithm;
import org.JIFSA.sensoryItems.Spatial2DObject;

import org.junit.Test;

/** Tests for the org.JIFSA.reasoning.distance.spatial2D.PolarDistanceAlgorithm class
 * 
 * @author Michael W. Floyd
 * @since 0.3
 */
public class PolarDistanceAlgorithmTest {

	
	
	/** Tests with a null first parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullFeature1() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		
		Spatial2DObject f = new Spatial2DObject("myFeaure");
		
		alg.pairwiseDistance(null,f);
	}
	
	/** Tests with a null first parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_nullFeature2() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		
		Spatial2DObject f = new Spatial2DObject("myFeaure");
		
		alg.pairwiseDistance(f,null);
	}
	
	/** Test with non-spatial features
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_bothNonSpatial() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		
		SensoryItem f1 = new SensoryItem("SensoryItem");
		SensoryItem f2 = new SensoryItem("SensoryItem");
		
		alg.pairwiseDistance(f1, f2);
	}
	
	/** Test with non-spatial features
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_firstNonSpatial() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		
		SensoryItem f1 = new SensoryItem("SensoryItem");
		Spatial2DObject f2 = new Spatial2DObject("SensoryItem");
		
		alg.pairwiseDistance(f1, f2);
	}
	
	/** Test with non-spatial features
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_secondNonSpatial() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		
		Spatial2DObject f1 = new Spatial2DObject("SensoryItem");
		SensoryItem f2 = new SensoryItem("SensoryItem");
		
		alg.pairwiseDistance(f1, f2);
	}
	

	
	/** Test a variety of valid pairs.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_validPairEqual() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		Spatial2DObject f1 = new Spatial2DObject("SensoryItem");
		Spatial2DObject f2 = new Spatial2DObject("SensoryItem");
		
		f1.setDistance(0f);
		f1.setDirection(0f);
		f2.setDistance(0f);
		f2.setDirection(0f);
		assertEquals(alg.pairwiseDistance(f1, f2),0f);
		
		f1.setDistance(9.5f);
		f1.setDirection(2.6f);
		f2.setDistance(9.5f);
		f2.setDirection(2.6f);
		assertEquals(alg.pairwiseDistance(f1, f2),0f);
		
		f1.setDistance(150f);
		f1.setDirection(-0.1f);
		f2.setDistance(150f);
		f2.setDirection(-0.1f);
		assertEquals(alg.pairwiseDistance(f1, f2),0f);
		
		f1.setDistance(-32f);
		f1.setDirection(-15f);
		f2.setDistance(-32f);
		f2.setDirection(-15f);
		assertEquals(alg.pairwiseDistance(f1, f2),0f);
	}
	
	/** Test a variety of valid pairs.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_validPairUnequal() {
		PolarDistanceAlgorithm alg = new PolarDistanceAlgorithm();
		Spatial2DObject f1 = new Spatial2DObject("SensoryItem");
		Spatial2DObject f2 = new Spatial2DObject("SensoryItem");
		
		f1.setDistance(0f);
		f1.setDirection(0f);
		f2.setDistance(10f);
		f2.setDirection(0f);
		assertEquals(alg.pairwiseDistance(f1,f2),10f);
		
		f1.setDistance(9.8f);
		f1.setDirection(0f);
		f2.setDistance(3.2f);
		f2.setDirection(0f);
		assertEquals(alg.pairwiseDistance(f1,f2),6.6f);
		
		f1.setDistance(-9.8f);
		f1.setDirection(0f);
		f2.setDistance(-3.2f);
		f2.setDirection(0f);
		assertEquals(alg.pairwiseDistance(f1,f2),6.6f);
		
		f1.setDistance(10f);
		f1.setDirection(-15f);
		f2.setDistance(10f);
		f2.setDirection(-45f);
		assertEquals(alg.pairwiseDistance(f1,f2),5.176381f);
		
		f1.setDistance(4f);
		f1.setDirection(23f);
		f2.setDistance(10f);
		f2.setDirection(85f);
		assertEquals(alg.pairwiseDistance(f1,f2),8.856765f);
	}

}
