/** Spatial2DObjectTest.java in the package tests.junit.org.JIFSA.sensoryItems of the JIFSA project.
    Originally created 29-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package tests.junit.org.JIFSA.sensoryItems;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.JIFSA.SensoryItem;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.spatial2D.PolarDistanceAlgorithm;
import org.JIFSA.sensoryItems.Spatial2DObject;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/** Tests for the Spacial2DObject class.
 * 
 * @author Michael W. Floyd
 * @since 0.2
 *
 */
public class Spatial2DObjectTest {

	/** Tests the constructor given a null feature name
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullName(){
		new Spatial2DObject(null);
	}
	
	/** Tests the constructor to make sure the default values
	 * of the distance and direction are set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_defaultValues(){
		Spatial2DObject s2D = new Spatial2DObject("2DFeature");
		assertEquals(s2D.getDirection(),0.0f);
		assertEquals(s2D.getDistance(),0.0f);
	}
	
	/** Tests the constructor to make sure the name is
	 * properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_nameSet(){
		Spatial2DObject s2D = new Spatial2DObject("myFeature");
		assertEquals(s2D.getFeatureName(),"myFeature");
	}
	
	/** Tests to see if the distance can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setDistance(){
		Spatial2DObject s2D = new Spatial2DObject("myFeaure");
		s2D.setDistance(9.9f);
		assertEquals(s2D.getDistance(),9.9f);
	}
	
	/** Tests to see if the direction can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setDirection(){
		Spatial2DObject s2D = new Spatial2DObject("myFeaure");
		s2D.setDirection(56.1f);
		assertEquals(s2D.getDirection(),56.1f);
	}
	
	/** Tests the equals method given a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_null(){
		Spatial2DObject s2D = new Spatial2DObject("myFeaure");
		assertFalse(s2D.equals(null));
	}
	
	/** Tests the equals method given a different object type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentClass(){
		Spatial2DObject s2D = new Spatial2DObject("myFeaure");
		String s = new String();
		assertFalse(s2D.equals(s));
	}
	
	/** Tests the equals method given two objects with
	 * the same name created with the default constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test 
	public void equals_default(){
		Spatial2DObject s2D1 = new Spatial2DObject("myFeaure");
		Spatial2DObject s2D2 = new Spatial2DObject("myFeaure");
		assertEquals(s2D1,s2D2);
	}
	
	/** Test the equality of two objects that do not
	 * have the default values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test 
	public void equals_nonDefault(){
		Spatial2DObject s2D1 = new Spatial2DObject("myFeaure");
		s2D1.setDistance(12.2f);
		s2D1.setDirection(0.04f);
		Spatial2DObject s2D2 = new Spatial2DObject("myFeaure");
		s2D2.setDistance(12.2f);
		s2D2.setDirection(0.04f);
		
		assertEquals(s2D1,s2D2);
	}
	
	/** Tests the equality of two objects with different
	 * feature names.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentName(){
		Spatial2DObject s2D1 = new Spatial2DObject("myFeaure");
		Spatial2DObject s2D2 = new Spatial2DObject("otherFeaure");
		assertFalse(s2D1.equals(s2D2));
	}
	
	/** Tests the equality of two objects with different
	 * directions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentDirection(){
		Spatial2DObject s2D1 = new Spatial2DObject("myFeaure");
		s2D1.setDistance(12.2f);
		s2D1.setDirection(9.3f);
		Spatial2DObject s2D2 = new Spatial2DObject("myFeaure");
		s2D2.setDistance(12.2f);
		s2D2.setDirection(1.1f);
		
		assertFalse(s2D1.equals(s2D2));
	}
	
	/** Tests the equality of two objects with different
	 * distances.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentDistance(){
		Spatial2DObject s2D1 = new Spatial2DObject("myFeaure");
		s2D1.setDistance(99.3f);
		s2D1.setDirection(0.04f);
		Spatial2DObject s2D2 = new Spatial2DObject("myFeaure");
		s2D2.setDistance(7.2f);
		s2D2.setDirection(0.04f);
		
		assertFalse(s2D1.equals(s2D2));
	}
	
	/** Tests the equality of an object to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test 
	public void equals_self(){
		Spatial2DObject s2d = new Spatial2DObject("2d");
		assertTrue(s2d.equals(s2d));
	}
	
	/** Tests the method with a null parameters.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setDistanceCalculation_null(){
		Spatial2DObject.setDistanceCalculation(null);
	}
	
	
	/** Provides the method with a null parameter
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_secondNull(){
		PolarDistanceAlgorithm pda = new PolarDistanceAlgorithm();
		Spatial2DObject.setDistanceCalculation(pda);
		Spatial2DObject f = new Spatial2DObject("feature");
		f.pairwiseDistance(null);
	}
	
	/** Tests if the distance calculation algorithm that
	 * has been set is correctly used.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_valid(){
		PolarDistanceAlgorithm pda = new PolarDistanceAlgorithm();
		Spatial2DObject.setDistanceCalculation(pda);
		
		Spatial2DObject f1 = new Spatial2DObject("feature");
		f1.setDistance(0.0f);
		f1.setDirection(0.0f);
		Spatial2DObject f2 = new Spatial2DObject("feature");
		f2.setDistance(7.2f);
		f2.setDirection(0.0f);
		
		assertEquals(f1.pairwiseDistance(f2), 7.2f);
	}
	
	/** Tests if it correctly uses the distance calculation of the 
	 * superclass if it does not have a disance calculation set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test
	public void pairwiseDistance_noAlgorithmSet(){
		EqualityDistanceAlgorithm eda = new EqualityDistanceAlgorithm();
		SensoryItem.setDistanceCalculation(eda);
		
		Spatial2DObject.removeDistanceCalculation();
		
		Spatial2DObject f1 = new Spatial2DObject("feature");
		f1.setDistance(0.0f);
		f1.setDirection(0.0f);
		Spatial2DObject f2 = new Spatial2DObject("feature");
		f2.setDistance(7.2f);
		f2.setDirection(0.0f);
		Spatial2DObject f3 = new Spatial2DObject("feature");
		f3.setDistance(0.0f);
		f3.setDirection(0.0f);
		
		assertEquals(f1.pairwiseDistance(f2), EqualityDistanceAlgorithm.MAXDISTANCE);
		assertEquals(f1.pairwiseDistance(f3), EqualityDistanceAlgorithm.MINDISTANCE);
		assertEquals(f3.pairwiseDistance(f2), EqualityDistanceAlgorithm.MAXDISTANCE);
	}
	
	/** Test giving a SensoryItem that isn't a Spatial2DObject
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_wrongType(){
		PolarDistanceAlgorithm pda = new PolarDistanceAlgorithm();
		Spatial2DObject.setDistanceCalculation(pda);
		
		SensoryItem f1 = new SensoryItem("feature");
		SensoryItem f2 = new Spatial2DObject("feature");
		
		f2.pairwiseDistance(f1);
	}
	
	/** Test giving a SensoryItem that isn't a Spatial2DObject
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_firstWrongType(){
		PolarDistanceAlgorithm pda = new PolarDistanceAlgorithm();
		Spatial2DObject.setDistanceCalculation(pda);
		
		SensoryItem f1 = new SensoryItem("feature");
		Spatial2DObject f2 = new Spatial2DObject("feature");
		
		f2.pairwiseDistance(f1);
	}
	
	/** Test giving a SensoryItem that isn't a Spatial2DObject
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	@Test(expected=IllegalArgumentException.class)
	public void pairwiseDistance_secondWrongType(){
		PolarDistanceAlgorithm pda = new PolarDistanceAlgorithm();
		Spatial2DObject.setDistanceCalculation(pda);
		
		Spatial2DObject f1 = new Spatial2DObject("feature");
		SensoryItem f2 = new SensoryItem("feature");
		
		f1.pairwiseDistance(f2);
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(Spatial2DObjectTest.class);
	}
	
}
