/** CaseBaseIOTest.java in the package tests.junit.org.JIFSA.tools of the JIFSA project.
    Originally created 18-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.JIFSA.tools;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.List;

import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.SensoryItem;
import org.JIFSA.tools.CaseBaseIO;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;


/** Unit tests for the org.JIFSA.tools.CaseBaseIO class
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class CaseBaseIOTest {

	private final String c_caseFile = "tests\\testdata\\testcasebase.cb";
	private final String c_otherObjType = "tests\\testdata\\otherObjType.cb";
	private final String c_invalidClass = "tests\\testdata\\invalidClas.cb";
	
	/** Test to make sure the a null filename parameter
	 * is caught by the loadCaseBase method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void loadCaseBase_nullFilename(){
		CaseBaseIO.loadCaseBase(null);
	}
	
	/** Create an object of type CaseBaseIO
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3 
	 *
	 */
	@Test
	public void testConstructor(){
		CaseBaseIO cbio = new CaseBaseIO();
		assertNotNull(cbio);
	}
	
	/** Test to make sure the a null filename parameter
	 * is caught by the saveCaseBase method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void saveCaseBase_nullFilename() throws Exception{
		CaseBase c = new CaseBase();
		CaseBaseIO.saveCaseBase(c,null);
	}
	
	/** Test to make sure the a null casebase parameter
	 * is caught by the saveCaseBase method
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void saveCaseBase_nullCaseBase() throws Exception{
		String s = "filename";
		CaseBaseIO.saveCaseBase(null,s);
	}
	
	/** Test to see if a case base is  outputed
	 * to a file and then loaded for the file
	 * if it is the same case base
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void writeReadAndCompare() throws Exception{
		Case c1 = new Case();
		
		AgentInputs v = new AgentInputs();
		SensoryItem f =  new SensoryItem("SensoryItem");
		v.addSensoryItem(f);
		List<AgentAction> l = new ArrayList<AgentAction>();
		AgentAction a1 = new AgentAction("Action1");
		AgentAction a2 = new AgentAction("Action2");
		l.add(a1);
		l.add(a2);
		Case c2 = new Case(v,l);
		
		CaseBase cb = new CaseBase();
		cb.addCase(c1);
		cb.addCase(c2);
		
		CaseBaseIO.saveCaseBase(cb, this.c_caseFile);
		
		CaseBase loaded = CaseBaseIO.loadCaseBase(this.c_caseFile);
		
		assertTrue(cb.equals(loaded));
	}
	
	
	/** Try loading a file that has another type of
	 * Object written to it.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void loadCaseBase_otherTypeOfClass(){
		try{
			//create the output streams
			FileOutputStream fileOut = new FileOutputStream(this.c_otherObjType);
			ObjectOutputStream oos = new ObjectOutputStream(fileOut);
	    	
			oos.writeObject(new String("not a casebase"));
			
			//close the output streams
			oos.close();
			fileOut.close();
		}catch(Exception e){
			assertTrue(false);
		}
		
		CaseBase loaded = CaseBaseIO.loadCaseBase(this.c_otherObjType);
		assertNull(loaded);
	}
	
	/** Try loading a file that has no Objects
	 * written to it.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void loadCaseBase_noObjects(){
		
		CaseBase loaded = CaseBaseIO.loadCaseBase(this.c_invalidClass);
		assertNull(loaded);
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(CaseBaseIOTest.class);
	}
}
