/** Brain.java in the package org.RCSImitate of the RCSImitate project.
    Originally created Mar 3, 2008

    Copyright (C) 2007 - 2008  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.RCSImitate;


import org.JIFSA.Agent;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.RCSImitate.soccersimactions.CatchAction;
import org.RCSImitate.soccersimactions.DashAction;
import org.RCSImitate.soccersimactions.KickAction;
import org.RCSImitate.soccersimactions.TurnAction;
import org.RCSImitate.soccersimactions.TurnNeckAction;

/** The brain of the imitative agent that will perform the CBR process.
 * 
 *
 * @author Michael W. Floyd
 * @since 0.3
 */
public class Brain extends Thread {

	//used to indicate when the game is over
	private volatile boolean m_timeOver;
	
	//the CBR agent
	private Agent m_agent;
	
	//the current inputs to the system
	private Vision m_vision;
	
	//the controll for the actual robot/agent
	private RCSImitate m_robot;
	
	/** Creates a Brain object given an agent (used to
	 * determine what actions to perform) and the 'body'
	 * of the robot (typically the RCSImitate object that
	 * created the Brain).
	 * 
	 * @param myAgent The agent that can determine actions to perform
	 * @param myRobot The body of the robot that can perform actions
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public Brain(Agent myAgent, RCSImitate myRobot){
		if(myAgent == null || myRobot == null){
			throw new IllegalArgumentException("Null value given to Brain constructor.");
		}
		this.m_timeOver = false;
		this.m_agent = myAgent;
		this.m_robot = myRobot;
		this.m_vision = new Vision();
	}
	
	/** The brain will run, constantly checking to see if any new
	 * AgentInputs have been sensed. Whenever a new AgentInput has
	 * been sensed (set using the see() method) it will use the
	 * supplied agent (in constructor) to determine an action to
	 * perform and perform that action.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 * 
	 * @Override
	 */
	@Override
	public void run(){
		//we loop until the game is over
		while( !this.m_timeOver ){
			
			//make sure we are not working to fast and are processing new info
			if(!this.m_vision.hasBeenSeen()){
				AgentInputs currentInput = this.m_vision.getCurrentVision();
				
				//now we perform the search
				AgentAction toPerform = this.m_agent.senseEnvironment(currentInput);
				
				//check for a null action, meaning no action found
				if(toPerform == null){
					System.out.println("No action returned. You may need to filter CaseBase to remove Cases with no actions!");
					continue;
				}
	
				String actionName = toPerform.getActionName();
				
				//now we perform the action
				if(actionName.equals(DashAction.c_DASH)){
					DashAction dash = (DashAction)toPerform;
					this.m_robot.dash(dash.getDashPower());
				}else if(actionName.equals(TurnAction.c_TURN)){
					TurnAction turn = (TurnAction)toPerform;
					this.m_robot.turn(turn.getTurnAngle());
				}else if(actionName.equals(KickAction.c_KICK)){
					KickAction kick = (KickAction)toPerform;
					this.m_robot.kick(kick.getKickPower(), kick.getKickAngle());
				}else if(actionName.equals(TurnNeckAction.c_TURNNECK)){
					TurnNeckAction turnneck = (TurnNeckAction)toPerform;
					this.m_robot.turn_neck(turnneck.getTurnNeckAngle());
				}else if(actionName.equals(CatchAction.c_CATCH)){
					CatchAction catchAct = (CatchAction)toPerform;
					//TODO catch
				}else{
					System.out.println("I was not expecting action: " + actionName);
				}
	
			}
		}

	}
	
	/** Used to store a new input to the agent to be processed.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public void see(AgentInputs inputs){
		  this.m_vision.store(inputs);
	  }

	/** Used to tell the brain that the game is over.
	 * 
	 * @param b True if the game is over, false otherwise
	 *
	 * @author Michael W. Floyd
	 * @since 0.3
	 */
	public void setTimeOver(boolean b) {
		this.m_timeOver = b;
	}
}
