/** RCSImitate.java in the package org.RCSImitate of the RCSImitate project.
    Originally created Mar 3, 2008

    Copyright (C) 2007 - 2008  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    * 
    */
package org.RCSImitate;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.StringTokenizer;

import javax.swing.SwingUtilities;

import org.JIFSA.Agent;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.tools.CaseEvent;
import org.JIFSA.tools.CaseEventListener;
import org.JIFSA.SensoryItem;
import org.JIFSA.sensoryItems.Spatial2DObject;
import org.JIFSA.preprocessing.filters.casebasefilter.NoActionsFilter;
import org.JIFSA.reasoning.Weights;
import org.JIFSA.reasoning.actionselection.ActionSelection;
import org.JIFSA.reasoning.actionselection.ClosestNeighbourSelection;
import org.JIFSA.reasoning.actionselection.actionestimation.ActionEstimation;
import org.JIFSA.reasoning.actionselection.actionestimation.LastActionEstimate;
import org.JIFSA.reasoning.casebasesearch.CaseBaseSearch;
import org.JIFSA.reasoning.casebasesearch.NearestNeighbourSearch;
import org.JIFSA.reasoning.distance.EqualityDistanceAlgorithm;
import org.JIFSA.reasoning.distance.GlobalDistanceMeasure;
import org.JIFSA.reasoning.distance.globaldistance.OrderIndexMatchingAlgorithm;
import org.JIFSA.reasoning.distance.penalty.ConstantPenalty;
import org.JIFSA.reasoning.distance.spatial2D.PolarDistanceAlgorithm;
import org.JIFSA.tools.CaseBaseIO;
import org.RCSImitate.casebasebuilder.LogFile2CaseBase;
import org.RCSImitate.gui.RCSIFrame;
import org.RCSImitate.sensoryItems.BallFeature;
import org.RCSImitate.sensoryItems.TeammatePlayerFeature;
import org.RCSImitate.parsingListeners.*;
import org.RCSLogServer.LogParser.*;

/** Executable class for the RCSImitate package. The class can either
 * be given the command line parameters to start an agent, or otherwise
 * a GUI will be loaded.
 * 
 * Note: Considerable portions of the communication code was taken
 * from the Krislet soccer agent written by Krzysztof Langner.
 * http://www.ida.liu.se/~frehe/RoboCup/Libs
 * 
 * @author Michael W. Floyd
 * @since 0.2
 * 
 */
public class RCSImitate implements Runnable, CaseEventListener {

    //the default parameters
    public static final String DEFAULT_HOSTNAME = "localhost";
    public static final int DEFAULT_HOSTPORT = 6000;
    public static final String DEFAULT_TEAMNAME = "Carleton";
    public static final String DEFAULT_CASEBASEFILE = "default.cb";
	
    //the minimum number of parameters the agent class needs
    private static final int MINNUMARGS = 2;
    //Size of socket buffer
    private static final int MSG_SIZE = 4096;
	
    //for communicating with the server
    private DatagramSocket m_socket;
    private InetAddress m_serverhost;
    private int m_serverport;
    private String m_teamname;
    private volatile boolean m_timeOver;
    private Brain m_brain;
    private Agent m_agent;
	
    /** The executable method.
     *
     * @author Michael W. Floyd
     * @since 0.2
     * 
     */
    public static void main(String[] args) {

	//if the user did not give enough command line arguments we start the GUI,
	// otherwise we start the agent
	if(args.length < RCSImitate.MINNUMARGS){
	    //we start the GUI to see what the user wants to do
	    SwingUtilities.invokeLater(new Runnable() {
		    public void run() {
			RCSIFrame inst = new RCSIFrame();
			inst.setLocationRelativeTo(null);
			inst.setVisible(true);
		    }
		});
	}else{
	    parseAndRun(args);
	}

    }
	
    /** Parses the command line arguments and initializes the agent.
     * 
     * @param args The command line arguments
     *
     * @author Michael W. Floyd
     * @since 0.3
     */
    private static void parseAndRun(String[] args){
	String hostName = RCSImitate.DEFAULT_HOSTNAME;
	int hostPort = RCSImitate.DEFAULT_HOSTPORT;
	String teamName = RCSImitate.DEFAULT_TEAMNAME;
	String cbFile = RCSImitate.DEFAULT_CASEBASEFILE;
		
	//TODO parse filters
	//TODO parse weights
	//TODO parse cbsearch type
	//TODO parse NN's
	//TODO parse distance calcs
	//TODO parse distance penalties
	//TODO parse case distance calc
	//TODO parse action selection
	//TODO parse action estimation
	//TODO parse penalty
		
	//parse any supplied arguments
	for(int ii=0; ii<args.length; ii += 2 ){
	    if( args[ii].compareTo("-host") == 0 ){
		hostName = args[ii+1];
	    }else if( args[ii].compareTo("-port") == 0 ){
		hostPort = Integer.parseInt(args[ii+1]);
	    }else if( args[ii].compareTo("-team") == 0 ){
		teamName = args[ii+1];
	    }else if( args[ii].compareTo("-casebasename") == 0){
		cbFile = args[ii+1];
	    }else{
		displayErrorMessage();
		return;
	    }
	}
		
	//TODO remove below
	System.out.println("Loading the Case Base ...");
	CaseBase casebase = CaseBaseIO.loadCaseBase(cbFile);
	NoActionsFilter naf = new NoActionsFilter();
	System.out.println("Filtering the Case Base ...");
	casebase = naf.filter(casebase);
	CaseBaseSearch cbSearch = new NearestNeighbourSearch(1);
		
	SensoryItem.setDistanceCalculation(new EqualityDistanceAlgorithm());
	SensoryItem.setPenaltyDistanceCalculation(new ConstantPenalty(100));
	Spatial2DObject.setDistanceCalculation(new PolarDistanceAlgorithm());
		
	Weights w = new Weights(0.0f);
	w.setWeight(BallFeature.c_BALL, 1.0f);
	w.setWeight(TeammatePlayerFeature.c_TEAMMATE, 1.0f);
	GlobalDistanceMeasure gd = new OrderIndexMatchingAlgorithm(w);
	Case.setGlobalDistanceCalculation(gd);

	ActionEstimation ae = new LastActionEstimate();
	ActionSelection actionSelection = new ClosestNeighbourSelection(ae); 
	
	Agent ag = new Agent(casebase,cbSearch,actionSelection);
	//TODO remove above
		
		
	//create the agent and run it
	RCSImitate agent = null;
	try {
	    agent = new RCSImitate(hostName,hostPort,teamName,ag);
	} catch (SocketException e) {
	    System.err.println(e.getMessage());
	    return;
	} catch (UnknownHostException e) {
	    System.err.println(e.getMessage());
	    return;
	}
		

	(new Thread(agent)).start();	
    }
	
    //TODO needs more args
    public RCSImitate(String serverhost, int serverport, String team, Agent myAgent) throws SocketException, UnknownHostException{
	//TODO verify the parameters
		
	this.m_socket = new DatagramSocket();
	this.m_timeOver = false;
	this.m_brain = null;
		
	//save the parameters that were given
	this.m_serverhost = InetAddress.getByName(serverhost);
	this.m_serverport = serverport;
	this.m_teamname = team;
	this.m_agent = myAgent;
		
    }
	
    /** Used to run the agent thread
     * 
     * @Override
     */
    public void run(){
	try {
	    execute();
	} catch (Exception e) {
	    System.out.println("Fatal Error: " + e.getMessage());
	}
    }
	
    /** Runs the RCSImitate agent.
     * 
     * @throws IOException
     *
     * @author Michael W. Floyd
     * @since 0.3
     */
    public void execute() throws IOException {
	byte[] buffer = new byte[RCSImitate.MSG_SIZE];
	DatagramPacket packet = new DatagramPacket(buffer, RCSImitate.MSG_SIZE);

	// first we need to initialize connection with server
	System.out.println("Sending init command to server...");
	init();

	//We initialize the parser
	LogParser parser=new LogParser();
	rcscenesParser PEListener= new rcscenesParser(m_teamname);
	PEListener.addCEListener((CaseEventListener) this);
	parser.addPEListener(PEListener); //register this listener with the logparser

	System.out.println("Waiting for response...");

	this.m_socket.receive(packet);
	String serverInit=new String(packet.getData());
	parser.parse(serverInit.trim());
	this.m_serverport = packet.getPort();

	System.out.println("Connection established.");

	//now we are connected and we can play

	while( this.m_timeOver != true ){
	    try{
		parser.parse(receive().trim());
	    } catch(IOException e){System.err.println(e.getMessage());
}
	}
		
	System.out.println("Game over. Shutting down.");
    }
	
    /** Used to move an agent to a specified location
     * 
     * @param x The x coordinate
     * @param y The y coordinate
     *
     * @since 0.3
     */
    public void move(double x, double y){
	//the message will be (move X Y)
	send("(move " + Double.toString(x) + " " + Double.toString(y) + ")");
    }

    /** Turns the agent by a specified moment
     * 
     * @param moment The amount to turn
     *
     * @since 0.3
     */
    public void turn(double moment){
	//the message will be (turn moment)
	send("(turn " + Double.toString(moment) + ")");
    }

    /** Turns the agent's neck by a specified moment
     * 
     * @param moment The amount to turn
     *
     * @since 0.3
     */
    public void turn_neck(double moment){
	//the message will be (turn_neck moment)
	send("(turn_neck " + Double.toString(moment) + ")");
    }

    /** The agent will dash with a specified power (speed)
     * 
     * @param power The dashing power
     *
     * @since 0.3
     */
    public void dash(double power){
	//the message will be (dash power)
	send("(dash " + Double.toString(power) + ")");
    }

    /** The agent will kick with a specified power and direction
     * 
     * @param power The kicking power
     * @param direction The direction to kick
     *
     * @since 0.3
     */
    public void kick(double power, double direction){
	//the message will be (kick power direction)
	send("(kick " + Double.toString(power) + " " + Double.toString(direction) + ")");
    }

    /** The agent will say something.
     * 
     * @param message The message to say
     *
     * @since 0.3
     */
    public void say(String message){
	//message will be (say message)
	send("(say " + message + ")");
    }

    /** Use to change the focus of the agent's view of the world.
     * 
     * @param angle The view angle (size of vision field)
     * @param quality The quality of view
     *
     * @since 0.3
     */
    public void changeView(String angle, String quality){
	//message will be (change_view angle quality)
	send("(change_view " + angle + " " + quality + ")");
    }

    /** Used by a goalie to catch the soccer ball.
     * 
     * @param direction The direction to catch
     *
     * @author Michael W. Floyd
     * @since 0.3
     */
    public void catchBall(String direction){
	//message will be (catch direction)
	send("(catch " + direction + ")");
    }
	
    /** Used to send the initialization command to the server.
     * 
     * @since 0.3
     */
    protected void init(){
	//the message is (init teamname (version 8))
        send("(init " + this.m_teamname + " (version 8))");
    }
	
    /** Used to sent the specified message to the server.
     * 
     * @param message The message to send
     *
     * @author Michael W. Floyd
     * @since 0.3
     */
    private void send(String message){
	byte[] buffer = new byte[RCSImitate.MSG_SIZE];
	buffer = message.getBytes();

	DatagramPacket packet 
	    = new DatagramPacket(buffer, buffer.length, this.m_serverhost, this.m_serverport);

	try{
	    this.m_socket.send(packet);
	}catch(IOException e){
	    System.err.println("Socket sending error : " + e);
	}
    }

    /** Used to receive the next message from the server.
     * 
     * @return The message received
     *
     * @author Michael W. Floyd
     * @since 0.3
     */
    private String receive() {
	byte[] buffer = new byte[RCSImitate.MSG_SIZE];
	DatagramPacket packet = new DatagramPacket(buffer, RCSImitate.MSG_SIZE);
	try{
	    this.m_socket.receive(packet);
	}catch(IOException e){
	    System.err.println("Socket receiving error : " + e);
	}
	return new String(buffer);
    }

    public void Connecting(CaseEvent ce){ }

    /** CaseEventListener method used to initialize the agent's Brain.
     * 
     * @param ce the Case Event.
     *
     * @author Edgar Acosta
     * @since 0.4
     */
    public void Connected(CaseEvent ce){ 
	// initialize player's brain
	this.m_brain = new Brain(this.m_agent,this);
		
	//start the Brain thread
	(new Thread(this.m_brain)).start();
    }

    public void TeamNameMissing(CaseEvent ce){ }


    /** CaseEventListener method used to pass the Inputs to the Brain.
     * 
     * @param ce the Case Event.
     *
     * @author Edgar Acosta
     * @since 0.4
     */
    public void GotInputs(CaseEvent ce)
    {
	AgentInputs ai = ce.getAgentInputs();
	this.m_brain.see(ai);
    }

    public void GotActions(CaseEvent ce){ }

    /** CaseEvent method used to stop the agent
     * 
     * @param ce the Case Event.
     *
     * @author Edgar Acosta
     * @since 0.4
     */
    public void TimeOver(CaseEvent ce){
	this.m_timeOver = true;
	this.m_brain.setTimeOver(true);
    }

    /** When a user gives invalid command line arguments this
     * method displays the correct arguments that can be given.
     * 
     * @author Michael W. Floyd
     * @since 0.3
     */
    private static void displayErrorMessage(){
	//XXX do something better
	System.out.println("Invalid params");
    }
}
