/** LogFile2CaseBase.java in the package org.RCSImitate.casebasebuilder of the RCSImitate project.
    Originally created 22-Jun-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    * 
    */

package org.RCSImitate.casebasebuilder;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.RCSLogServer.LogParser.*;
import org.RCSImitate.parsingListeners.*;
import org.RCSImitate.sensoryItems.BallFeature;
import org.RCSImitate.sensoryItems.FlagFeature;
import org.RCSImitate.sensoryItems.GoalFeature;
import org.RCSImitate.sensoryItems.LineFeature;
import org.RCSImitate.sensoryItems.OpponentPlayerFeature;
import org.RCSImitate.sensoryItems.PlayerFeature;
import org.RCSImitate.sensoryItems.RoboCupSimulation2DSpatialObject;
import org.RCSImitate.sensoryItems.TeammatePlayerFeature;
import org.RCSImitate.sensoryItems.UnknownPlayerFeature;
import org.RCSImitate.soccersimactions.CatchAction;
import org.RCSImitate.soccersimactions.DashAction;
import org.RCSImitate.soccersimactions.KickAction;
import org.RCSImitate.soccersimactions.TurnAction;
import org.RCSImitate.soccersimactions.TurnNeckAction;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.tools.CaseBaseIO;
import org.JIFSA.tools.CaseEvent;
import org.JIFSA.tools.CaseEventListener;



/** Used to create a case-base from the visual input/ action
 * of a RoboCup Simulation League agent.
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class LogFile2CaseBase implements CaseEventListener
{

    private String m_logFileName;
    private String m_outputFileName;
    private CaseBase m_casebase;
    private String m_teamName;
    private AgentInputs m_lastInputs;
    private ArrayList<AgentAction> m_actionList;
    
    
    /** This will in-take the command line parameters from the
     * user and perform the LogFile to CaseBase conversion.
     * 
     * @param args Command line arguments
     *
     * @author Michael W. Floyd
     * @since 0.2
     */
    public static void main(String[] args) {
	// There should be two parameters. We ignore anything else.
	if (args.length < 2)
	    {
		displayUsage();
		return;
	    }
	
	String inFile = args[0];
	String outFile = args[1];
	String teamName = "";
	if(args.length >2){
	    teamName = args[2];
	}
	
	//we will create an instance of the class
	LogFile2CaseBase lf2cb = new LogFile2CaseBase(inFile, outFile, teamName);
	
	//parse the log file and create the case base
	try{
	    lf2cb.parseLogFile();
	}catch(IOException e){
	    System.err.println("Error parsing log file: " + e.getMessage());
	    return;
	}
	
	System.out.println("Writing Cases to file...");
	//write the case base to a file
	try{
	    lf2cb.writeCaseBaseFile();
	}catch(IOException e){
	    System.err.println("Error writing to output file: " + e.getMessage());
	}
	System.out.println("Finished!");
    }
    
    
    /** The constructor for the LogFile2CaseBase class. The
     * name of the input logfile, the name of the file
     * to save the case base to and the players team name
     * are provided by the user. 
     * 
     * @param logFile The name of the logfile to get the date from
     * @param caseBaseFile The name of the file to write the casebase to
     * @param teamName The name of the team the player is on
     * 
     * @author Michael W. Floyd
     * @since 0.2
     */
    public LogFile2CaseBase(String logFile, String caseBaseFile, String teamName) {
	//check params
	if(logFile == null || caseBaseFile == null || teamName == null){
	    throw new IllegalArgumentException("Null parameters given to constructor.");
	}
	
	//save the name of the files to read and write to and the team name
	this.m_logFileName = logFile;
	this.m_outputFileName = caseBaseFile;
	this.m_teamName = teamName;
	
	
	//create a null AgentInputs, since no messages have been processed yet and an empty AgentAction list
	m_lastInputs = new AgentInputs();
	m_actionList = new ArrayList<AgentAction>();

	//initialize the CaseBase
	this.m_casebase= new CaseBase();
    }

    /** CaseEventListener method called when in the log file it is found an agent init message. It is used here to fetch the team name
     *
     * @param ce the CaseEvent. It contains the team name
     *
     * @author Edgar Acosta
     * @since 0.4
     *
     */
    public void Connecting(CaseEvent ce){
	m_teamName=ce.getTeamName();
	System.out.println("Got team name from the Log File: '"+m_teamName+"'");
    }

    public void Connected(CaseEvent ce){ }

    /** CaseEventListener method called when we don't know the team name.
     *
     * @param ce the CaseEvent.
     *
     * @author Edgar Acosta
     * @since 0.4
     *
     */
    public void TeamNameMissing(CaseEvent ce){
	System.err.println("The Log File doesn't contain initialization messages.");
	System.err.println("Please run again with the third parameter.");
	System.err.println(" ");
	displayUsage();
	System.exit(1);
    }

    /** CaseEventListener method called when the Case Inputs have been parsed.
     *
     * @param ce the CaseEvent. It contains the Case Inputs
     *
     * @author Edgar Acosta
     * @since 0.4
     *
     */
    public void GotInputs(CaseEvent ce){
	m_lastInputs=ce.getAgentInputs();
    }

    /** CaseEventListener method called when the Case Outputs have been parsed.
     *
     * @param ce the CaseEvent. It contains the Case Outputs
     *
     * @author Edgar Acosta
     * @since 0.4
     *
     */
    public void GotActions(CaseEvent ce){
	m_actionList=ce.getAgentActions();
	Case c = new Case(m_lastInputs,m_actionList);
	m_casebase.addCase(c);
    }

    public void TimeOver(CaseEvent ce){}

    /** This method loads the log file and feeds it to the
     * LogParser. The LogParser will generate events that will be
     * catched by this listener.
     * 
     * @author Edgar Acosta
     * @since 0.2
     */
    public void parseLogFile() throws IOException {
	//open the log file
	BufferedReader in = new BufferedReader(new FileReader(this.m_logFileName));
	LogParser parser=new LogParser();
	rcscenesParser PEListener= new rcscenesParser(m_teamName);
	PEListener.addCEListener((CaseEventListener) this);
	
	parser.addPEListener(PEListener); //register this listener with the logparser
	
	//the next line in the log file
	String Line;
	//the parse line. The parser now uses the event model, so this will disapear in the future
	ParsedLine parsed=null;
	
	System.out.println("Reading log data...");
	//read in each line of the log file
	while ((Line = in.readLine()) != null){
	    //parse the line
	    parsed=parser.parse(Line);
	}
	PEListener.EoF();
	System.out.println("Finished reading log file.");
	System.out.println(this.m_casebase.getCasebaseSize() + " Cases were extracted.");
	
	//close the file input stream
	in.close();
    }



    /** Returns a copy of the case base created from the
     * log file.
     * 
     * @return The case base
     *
     * @author Michael W. Floyd
     * @since 0.2
     */
    public CaseBase getCaseBase(){
	return new CaseBase(this.m_casebase);
    }
	
    /** Saves the case base into the specified file
     * 
     *
     * @author Michael W. Floyd
     * @since 0.2
     */
    public void writeCaseBaseFile() throws IOException {
		
	CaseBaseIO.saveCaseBase(this.m_casebase,this.m_outputFileName);
		
    }

    /** Used to display usage instructions    
     * 
     * @author Edgar Acosta
     * @since 0.4
     */
    private static void displayUsage(){
	System.err.println("LogFile2CaseBase - converts captured Robocup log files to a CaseBase");
	System.err.println("\nUsage: java LogFile2CaseBase in-file out-file teamName");
	System.err.println("in-file: The name of the log file");
	System.err.println("out-file: The name of the file the CaseBase will be writen to");
	System.err.println("teamName: The name of the team the player was on when the log file was created.");
	System.err.println("          Only required if the Log File doesn't start with (init) messages.");
    }
}
