/** rcscenesParser.java in the package org.RCSImitate.parsingListener of the RCSImitate project.
    Originally created 16-Oct-2008

    Copyright (C) 2008  Edgar Acosta

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package org.RCSImitate.parsingListeners;

import java.util.ArrayList;

import org.RCSLogServer.LogParser.*;

import org.RCSImitate.sensoryItems.BallFeature;
import org.RCSImitate.sensoryItems.FlagFeature;
import org.RCSImitate.sensoryItems.GoalFeature;
import org.RCSImitate.sensoryItems.LineFeature;
import org.RCSImitate.sensoryItems.OpponentPlayerFeature;
import org.RCSImitate.sensoryItems.PlayerFeature;
import org.RCSImitate.sensoryItems.RoboCupSimulation2DSpatialObject;
import org.RCSImitate.sensoryItems.TeammatePlayerFeature;
import org.RCSImitate.sensoryItems.UnknownPlayerFeature;
import org.RCSImitate.soccersimactions.CatchAction;
import org.RCSImitate.soccersimactions.DashAction;
import org.RCSImitate.soccersimactions.KickAction;
import org.RCSImitate.soccersimactions.TurnAction;
import org.RCSImitate.soccersimactions.TurnNeckAction;
import org.JIFSA.AgentAction;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.tools.CaseBaseIO;
import org.JIFSA.tools.CaseEvent;
import org.JIFSA.tools.CaseEventListener;

/** This class is the ParsingEventListener that uses the parsing
 * Events to generate the same Cases as the first version of RCSImitate
 * (it used to use a different parser). The name comes from the fact that
 * RCSImitate is the succesor of rcscenes, and there are other possible
 * Case definitions.
 *
 * @author Edgar Acosta
 * @since 0.4
 *
 */
public class rcscenesParser extends ParsingEventListenerAdapter {

    private AgentInputs m_lastInputs;
    private ArrayList<AgentAction> m_actionList;
    private String m_teamName;
    private ArrayList<CaseEventListener> CEListeners;

    /** The constructor for the LogFile2CaseBase class. The
     * name of the input logfile, the name of the file
     * to save the case base to and the players team name
     * are provided by the user. 
     * 
     * @param teamName The name of the team the player is on
     * 
     * @author Edgar Acosta
     * @since 0.4
     */
    public rcscenesParser(String teamName) {
	m_teamName = teamName;
	
	
	//create a null AgentInputs, since no messages have been processed yet and an empty AgentAction list
	m_lastInputs = new AgentInputs();
	m_actionList = new ArrayList<AgentAction>();

	//create the list of CaseEventListeners
	CEListeners = new ArrayList<CaseEventListener>();
    }

    /** Registers a new CaseEventListener.
     *
     * @param cel The CaseEventListener
     *
     * @author Edgar Acosta
     *
     */
    public void addCEListener(CaseEventListener cel){
	CEListeners.add(cel);
    }

    /** Registers a CaseEventListener.
     *
     * @param cel The CaseEventListener
     *
     * @author Edgar Acosta
     *
     */
    public void removeCEListener(CaseEventListener cel){
	CEListeners.remove(cel);
    }

    /** ParsingEvent that occurs when the client sends an init message.
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void ClientInit(ParsingEvent pe){
	m_teamName=pe.get("team_name");
	CaseEvent ce = new CaseEvent(this,m_teamName);
	for(CaseEventListener cel:CEListeners) cel.Connecting(ce);
    }

    /** ParsingEvent that occurs when the server acknowledges an init message.
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void ServerInit(ParsingEvent pe){
	CaseEvent ce = new CaseEvent(this);
	for(CaseEventListener cel:CEListeners) cel.Connected(ce);
    }

    /** Parsing Event that occurs when the agent start receiving sensor
     * messages after being sending control messages.
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Sensing(ParsingEvent pe){
	if(m_teamName.equals("")){
	    CaseEvent ce = new CaseEvent(this);
	    for(CaseEventListener cel:CEListeners) cel.TeamNameMissing(ce);
	}
    }

    /** Parsing Event that occurs when a see message has been received
     * (before parsing it). Other events will process the objects in
     * the field.
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void SeeReceived(ParsingEvent pe){
	if(m_lastInputs.getNumberSensoryItems()>0){
	    CaseEvent ce = new CaseEvent(this,m_actionList);
	    for(CaseEventListener cel:CEListeners) cel.GotActions(ce);
	}
	m_lastInputs = new AgentInputs();
	m_actionList = new ArrayList<AgentAction>();
    }

    /** This is a method that signals we are at the end of the log file
     *
     * @author Edgar Acosta
     *
     */
    public void EoF(){
	if(m_lastInputs.getNumberSensoryItems()>0){
	    CaseEvent ce = new CaseEvent(this,m_actionList);
	    for(CaseEventListener cel:CEListeners) cel.GotActions(ce);
	}
    }

    /** Parsing Event that occurs when a see message has been parsed.
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void SeeParsed(ParsingEvent pe){
	    CaseEvent ce = new CaseEvent(this,m_lastInputs);
	    for(CaseEventListener cel:CEListeners) cel.GotInputs(ce);
    }

    /** Generic method to get feature values from a Parsing Event and
     * copy them to a RoboCupSimulation2DSpatialObject.
     *
     * @param so the 2D Spatial Object receiving the feature values
     * @param pe the Parsing Event containing the feature values
     *
     * @author Edgar Acosta
     *
     */
    private void setObjectPosition(RoboCupSimulation2DSpatialObject so,
			    ParsingEvent pe){//this sets the parameters that are common to all field objects
	if(pe.contains("dir_change")) so.setDirectionChange(Float.parseFloat(pe.get("dir_change")));
	if(pe.contains("dist_change")) so.setDistanceChange(Float.parseFloat(pe.get("dist_change")));
	if(pe.contains("direction")) so.setDirection(Float.parseFloat(pe.get("direction")));
	if(pe.contains("distance")) so.setDistance(Float.parseFloat(pe.get("distance")));
    }

    /** Parsing Event that occurs when a player has been sensed
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Player(ParsingEvent pe){
	PlayerFeature pf=null;
	if(pe.contains("team")){
	    String team=pe.get("team");
	    if(team.equals(m_teamName))
		pf = new TeammatePlayerFeature();
	    else
		pf = new OpponentPlayerFeature();
	} else pf= new UnknownPlayerFeature();
	if(pe.contains("player_number")) pf.setUniformNumber(Integer.parseInt(pe.get("player_number")));
	if(pe.contains("goalie")) pf.setIsGoalie(true);
// 	if(pe.contains("head_dir")) pf.set(Float.parseFloat(pe.get("head_dir"))); //these are ignored on this implementation
// 	if(pe.contains("body_dir")) pf.set(Float.parseFloat(pe.get("body_dir")));
	setObjectPosition(pf,pe);
	m_lastInputs.addSensoryItem(pf);
    }

    /** Parsing Event that occurs when a goal has been sensed
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Goal(ParsingEvent pe){
	GoalFeature gf = new GoalFeature();
	if(pe.contains("name"))
	    gf.setSide(pe.get("name"));
	setObjectPosition(gf,pe);
	m_lastInputs.addSensoryItem(gf);
    }

    /** Parsing Event that occurs when a ball has been sensed
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Ball(ParsingEvent pe){
	BallFeature bf = new BallFeature();
	setObjectPosition(bf,pe);
	m_lastInputs.addSensoryItem(bf);
    }

    /** Parsing Event that occurs when a Flag has been sensed
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Flag(ParsingEvent pe){
	FlagFeature flag = new FlagFeature();
	String name=pe.get("name");
	if(pe.contains("Box")) flag.setBox(pe.get("Box"));
	if(pe.contains("hp")) 
	    flag.setHorizontalPosition(pe.get("hp"));
	if(pe.contains("vp")) 
	    flag.setVerticalPosition(pe.get("vp"));
	if(pe.contains("flag_number")){
	    flag.setFlagNumber(Integer.parseInt(pe.get("flag_number")));
	    flag.setOutOfBounds(true);
	} else
	    flag.setOutOfBounds(false);
	setObjectPosition(flag,pe);
	m_lastInputs.addSensoryItem(flag);
    }

    /** Parsing Event that occurs when a line has been sensed
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Line(ParsingEvent pe){
	LineFeature lf = new LineFeature();
	if(pe.contains("name")) lf.setLocation(pe.get("name"));
	setObjectPosition(lf,pe);
	m_lastInputs.addSensoryItem(lf);
    }

    /* Actions */

    /** Parsing Event that occurs when a catch action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Catch(ParsingEvent pe){
	CatchAction catchAct = new CatchAction();
	m_actionList.add(catchAct);
    }

    /** Parsing Event that occurs when a dash action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Dash(ParsingEvent pe){
	float power =Float.parseFloat(pe.get("power"));
	DashAction dash = new DashAction(power);
	m_actionList.add(dash);
    }

    /** Parsing Event that occurs when a kick action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Kick(ParsingEvent pe){
	float power =Float.parseFloat(pe.get("power"));
	float angle =Float.parseFloat(pe.get("direction"));
	KickAction kick = new KickAction(power,angle);
	m_actionList.add(kick);
    }

    /** Parsing Event that occurs when a turn action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Turn(ParsingEvent pe){
	float angle =Float.parseFloat(pe.get("direction"));
	TurnAction turn = new TurnAction(angle);
	m_actionList.add(turn);
    }

    /** Parsing Event that occurs when a turn neck action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void TurnNeck(ParsingEvent pe){
	float angle =Float.parseFloat(pe.get("neck_angle"));
	TurnNeckAction turnNeck = new TurnNeckAction(angle);
	m_actionList.add(turnNeck);
    }

    /** Parsing Event that occurs when a hear action has been uttered
     *
     * @param pe The ParsingEvent
     *
     * @author Edgar Acosta
     *
     */
    public void Hear(ParsingEvent pe){
	String who=pe.get("who");
	String what=pe.get("what");
	if(who.equals("referee") && what.equals("time_over")){
	    CaseEvent ce = new CaseEvent(this);
	    for(CaseEventListener cel:CEListeners) cel.TimeOver(ce);
	}	    
    }

}
