/** FlagFeature.java in the package org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.RCSImitate.sensoryItems;

/** Represents a flag in a RoboCup Simulation soccer game
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class FlagFeature extends RoboCupSimulation2DSpatialObject {

	private static final long serialVersionUID = 3877696748585175328L;

	public static String c_FLAG = "RoboCupSimulation.Flag";
	
	//represent which box the flags are on
	public static String c_NO_BOX = "No Box";
	public static String c_PENALTY_BOX = "p";
	public static String c_GOAL_BOX = "g";
	
	//when the flag has no flag number
	public static final int c_NO_NUMBER = -1;
	
	//what location (left/right/center/top/bottom) on the field the flag is at
	public static String c_UNKNOWN_POSITION = "Unknown";
	public static String c_LEFT = "l";
	public static String c_RIGHT = "r";
	public static String c_CENTER = "c";
	public static String c_TOP = "t";
	public static String c_BOTTOM = "b";
	
	
	//member variables
	private boolean m_outOfBounds;
	private int m_flagNumber;
	private String m_box;
	private String m_horizontal;
	private String m_vertical;
	
	/** Creates a flag feature. By default the flag is
	 * said to be in bounds, has no flag number, is not
	 * located on a box and has an unknown vertical and
	 * horizontal position.
	 *  
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public FlagFeature(){
		super(FlagFeature.c_FLAG);
		this.m_outOfBounds = false;
		this.m_flagNumber = FlagFeature.c_NO_NUMBER;
		this.m_box = FlagFeature.c_NO_BOX;
		this.m_horizontal = FlagFeature.c_UNKNOWN_POSITION;
		this.m_vertical = FlagFeature.c_UNKNOWN_POSITION;
	}
	
	/** Sets the horizonal position of the flag. Valid values
	 * are the c_LEFT, c_RIGHT and c_CENTER constants.
	 * 
	 * @param pos The horizontal position of the flag.
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setHorizontalPosition(String pos){
		//check params
		if(pos == null){
			throw new IllegalArgumentException("Horizonal position of a flag set to a null value.");
		}
		
		//make sure a valid value was given
		if(!pos.equals(FlagFeature.c_LEFT) && !pos.equals(FlagFeature.c_RIGHT) && !pos.equals(FlagFeature.c_CENTER) && !pos.equals(FlagFeature.c_TOP) && !pos.equals(FlagFeature.c_BOTTOM)){
			throw new IllegalArgumentException("Horizonal position of a flag set to an invalid value: " + pos);
		}
		
		this.m_horizontal = pos;
	}
	
	/** Returns the horizontal position of the flag. This
	 * will be one of the following constants: c_LEFT, 
	 * c_RIGHT, c_CENTER or c_UNKNOWN_POSITION
	 * 
	 * @return The horizontal position
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public String getHorizontalPosition(){
		return this.m_horizontal;
	}
	
	/** Sets the vertical position of the flag. Valid values
	 * are the c_TOP, c_BOTTOM and c_CENTER constants.
	 * 
	 * @param pos The vertical position of the flag.
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setVerticalPosition(String pos){
		//check params
		if(pos == null){
			throw new IllegalArgumentException("Vertical position of a flag set to a null value.");
		}
		
		//make sure a valid value was given
		if(!pos.equals(FlagFeature.c_TOP) && !pos.equals(FlagFeature.c_BOTTOM) && !pos.equals(FlagFeature.c_CENTER) && !pos.equals(FlagFeature.c_LEFT) && !pos.equals(FlagFeature.c_RIGHT)){
			throw new IllegalArgumentException("Vertical position of a flag set to an invalid value: " + pos);
		}
		
		this.m_vertical = pos;
	}
	
	/** Returns the vertical position of the flag. This
	 * will be one of the following constants: c_TOP, 
	 * c_BOTTOM, c_CENTER or c_UNKNOWN_POSITION
	 * 
	 * @return The horizontal position
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public String getVerticalPosition(){
		return this.m_vertical;
	}
	
	/** Sets if the flag is located out of bounds.
	 * 
	 * @param out True if the flag is out of bounds.
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setOutOfBounds(boolean out){
		this.m_outOfBounds = out;
	}
	
	/** Returns true if the flag is out of bounds.
	 * 
	 * @return If the flag is out of bounds.
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public boolean isOutOfBounds(){
		return this.m_outOfBounds;
	}
	
	/** Sets the number of the flag.
	 * 
	 * @param num The number of the flag
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setFlagNumber(int num){
		this.m_flagNumber = num;
	}
	
	/** Returns the number of the flag.
	 * 
	 * @return The flag number
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public int getFlagNumber(){
		return this.m_flagNumber;
	}
	
	/** Sets the box the flag is on. Valid values
	 * are the c_GOAL_BOX or c_PENALTY_BOX.
	 * 
	 * @param pos The box the flag is on.
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setBox(String box){
		//check params
		if(box == null){
			throw new IllegalArgumentException("Box location of a flag set to a null value.");
		}
		
		//make sure a valid value was given
		if(!box.equals(FlagFeature.c_GOAL_BOX) && !box.equals(FlagFeature.c_PENALTY_BOX) ){
			throw new IllegalArgumentException("Box location of a flag set to an invalid value: " + box);
		}
		
		this.m_box = box;
	}
	
	/** Returns the box the flag is on. This
	 * will be one of the following constants: c_GOAL_BOX, 
	 * c_PENALTY_BOX or c_NO_BOX
	 * 
	 * @return The box the flag is on
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public String getBox(){
		return this.m_box;
	}
	
	/** Overrides the equals method
	 * 
	 */
	@Override
	public boolean equals(Object o){
		//check if they are the same exact object
		if(this == o){
			return true;
		}
		//check for null or wrong object type
		if(o == null || !(o instanceof FlagFeature)){
			return false;
		}
		
		//typecast object
		FlagFeature ff = (FlagFeature)o;
		
		//compare super class attributes
		if(!super.equals(ff)){
			return false;
		}
		
		//compare local attributes
		return ( (this.m_outOfBounds == ff.isOutOfBounds()) && 
				  (this.m_flagNumber == ff.getFlagNumber()) && 
				  (this.m_horizontal.equals(ff.getHorizontalPosition())) && 
				  (this.m_vertical.equals(ff.getVerticalPosition())) && 
				  (this.m_box.equals(ff.getBox())));
	}
}
