/** GoalFeature.java in the package org.RCSImitate.sensoryItems of the JIFSA project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.RCSImitate.sensoryItems;

/** Represents a goal net in RoboCup Simulation soccer.
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class GoalFeature extends RoboCupSimulation2DSpatialObject {

	private static final long serialVersionUID = -2415392647536412278L;
	
	//used to hold the values that mean the goal is the left or right goal
	public static String c_SIDE_LEFT = "l";
	public static String c_SIDE_RIGHT = "r";
	public static String c_SIDE_UNKNOWN = "UNKNOWN";
	
	public static String c_GOAL = "RoboCupSimulation.Goal";
	
	private String m_side;
	
	/** Construct a GoalFeature that is on an unknown side.
	 * 
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public GoalFeature(){
		super(GoalFeature.c_GOAL);
		this.m_side = GoalFeature.c_SIDE_UNKNOWN;
	}
	
	/** Set the side to either "l" (left) or "r" (right).
	 * 
	 * @param side The side the net is on
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setSide(String side){
		//test the params
		if(side == null){
			throw new IllegalArgumentException("Null parameter given.");
		}
		//make sure a valid side selected
		String lowerside = side.toLowerCase();
		if( !(lowerside.equals(GoalFeature.c_SIDE_LEFT)) && !(lowerside.equals(GoalFeature.c_SIDE_RIGHT))){
			throw new IllegalArgumentException("Invalid side given for goal.");
		}
		
		//save the side
		this.m_side = lowerside;
	}
	
	/** Return the side the GoalFeature is on
	 * 
	 * @return The side the GoalFeature is on
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public String getSide(){
		return new String(this.m_side);
	}
	
	/** Overrides the equals method
	 * 
	 */
	@Override
	public boolean equals(Object o){
		//check for same object
		if(this == o){
			return true;
		}
		//check for null or other object type
		if(o == null || !(o instanceof GoalFeature)){
			return false;
		}
		
		//typecast object
		GoalFeature gf = (GoalFeature)o;
		
		//see if the super class attributes are equal
		if(!super.equals(o)){
			return false;
		}
		
		return this.m_side.equals(gf.getSide());
	}
}
